/*
 * Copyright (c) 2003 World Wide Web Consortium,
 *
 * (Massachusetts Institute of Technology, European Research Consortium for
 * Informatics and Mathematics, Keio University). All Rights Reserved. This
 * work is distributed under the W3C(r) Software License [1] in the hope that
 * it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * [1] http://www.w3.org/Consortium/Legal/2002/copyright-software-20021231
 */

package org.w3c.dom.events;

import org.w3c.dom.views.AbstractView;

/**
 *  The <code>KeyboardEvent</code> interface provides specific contextual 
 * information associated with keyboard devices. Each keyboard event 
 * reference a key using an identifier. 
 * <p> Each modifier attribute (<code>ctrlKey</code>, <code>shiftKey</code>, 
 * <code>altKey</code>, <code>metaKey</code>, and <code>altGraphKey</code>) 
 * is activated when the key modifier is being pressed down or maintained 
 * pressed, i.e. the modifier attribute is not in use when the key modifier 
 * is being released.  To create an instance of the 
 * <code>KeyboardEvent</code> interface, use the feature string 
 * "KeyboardEvent" as the value of the input parameter used with the 
 * <code>DocumentEvent.createEvent</code> method. 
 * <p>See also the <a href='http://www.w3.org/TR/2003/WD-DOM-Level-3-Events-20030221'>Document Object Model (DOM) Level 3 Events Specification</a>.
 * @since DOM Level 3
 */
public interface KeyboardEvent extends UIEvent {
    // KeyLocationCode
    /**
     *  The key activation is not distinguished as the left or right version 
     * of the key, and did not originate from the numeric keypad (or did not 
     * originate with a virtual key corresponding to the numeric keypad). 
     * Example: the 'Q' key on a PC 101 Key US. 
     */
    public static final int DOM_KEY_LOCATION_STANDARD = 0x00;
    /**
     *  The key activated is in the left key location (there is more than one 
     * possible location for this key). Example: the left Shift key on a PC 
     * 101 Key US. 
     */
    public static final int DOM_KEY_LOCATION_LEFT     = 0x01;
    /**
     *  The key activation is in the right key location (there is more than 
     * one possible location for this key). Example: the right Shift key on 
     * a PC 101 Key US. 
     */
    public static final int DOM_KEY_LOCATION_RIGHT    = 0x02;
    /**
     *  The key activation originated on the numeric keypad or with a virtual 
     * key corresponding to the numeric keypad. Example: the '1' key on a PC 
     * 101 Key US. 
     */
    public static final int DOM_KEY_LOCATION_NUMPAD   = 0x03;
    /**
     *  Implementations can use this constant to indicate that the location of 
     * the key cannot be determined.  In case a DOM implementation wishes to 
     * provide new location information, all values above the value of this 
     * constant can be used for that effect and generic DOM applications 
     * should consider values above the value of this constant as being 
     * equivalent to <code>DOM_KEY_LOCATION_UNKNOWN</code>. 
     */
    public static final int DOM_KEY_LOCATION_UNKNOWN  = 0x04;

    /**
     *  <code>keyIdentifier</code> holds the identifier of the key. For a list 
     * of possible value, refer to .  Implementations that are unable to 
     * identify a key must use the key identifier <code>"Unidentified"</code>
     * . 
     */
    public String getKeyIdentifier();

    /**
     *  The <code>keyLocation</code> attribute contains an indication of the 
     * location of they key on the device. 
     */
    public int getKeyLocation();

    /**
     *  <code>true</code> if the control (Ctrl) key modifier is activated. 
     */
    public boolean getCtrlKey();

    /**
     *  <code>true</code> if the shift (Shift) key modifier is activated. 
     */
    public boolean getShiftKey();

    /**
     *  <code>true</code> if the alt (alternative) key modifier is activated. 
     */
    public boolean getAltKey();

    /**
     *  <code>true</code> if the meta (Meta) key modifier is activated.  The 
     * Command key modifier on Macintosh system must be represented using 
     * this key modifier. 
     */
    public boolean getMetaKey();

    /**
     *  <code>true</code> if the alt-graph (Alt Gr) key modifier is activated. 
     *  Some operating systems simulate the alt-graph key modifier with the 
     * combinaison of alt and ctrl key modifiers. Implementations are 
     * encouraged to use this modifier instead. 
     */
    public boolean getAltGraphKey();

    /**
     *  The <code>initKeyboardEvent</code> method is used to initialize the 
     * value of a <code>KeyboardEvent</code> created using the 
     * <code>DocumentEvent.createEvent</code> method. This method may only 
     * be called before the <code>KeyboardEvent</code> has been dispatched 
     * via the <code>EventTarget.dispatchEvent</code> method, though it may 
     * be called multiple times before being dispatched if necessary. If 
     * called multiple times, the final invocation takes precedence. This 
     * method has no effect if called after the event has been dispatched. 
     * @param typeArg Specifies the event type.
     * @param canBubbleArg Specifies whether or not the event can bubble. 
     *   This parameter overrides the intrinsic bubbling behavior of the 
     *   event.
     * @param cancelableArg Specifies whether or not the event's default 
     *   action can be prevent. This parameter overrides the intrinsic 
     *   cancelable behavior of the event.
     * @param viewArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>AbstractView</code>.
     * @param keyIdentifierArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>keyIdentifier</code> attribute.
     * @param keyLocationArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>keyLocation</code> attribute.
     * @param ctrlKeyArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>ctrlKey</code> attribute.
     * @param shiftKeyArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>shiftKey</code> attribute.
     * @param altKeyArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>altKey</code> attribute.
     * @param metaKeyArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>metaKey</code> attribute.
     * @param altGraphKeyArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>altGraphKey</code> attribute.
     */
    public void initKeyboardEvent(String typeArg, 
                                  boolean canBubbleArg, 
                                  boolean cancelableArg, 
                                  AbstractView viewArg, 
                                  String keyIdentifierArg, 
                                  int keyLocationArg, 
                                  boolean ctrlKeyArg, 
                                  boolean shiftKeyArg, 
                                  boolean altKeyArg, 
                                  boolean metaKeyArg, 
                                  boolean altGraphKeyArg);

    /**
     * The <code>initKeyboardEventNS</code> method is used to initialize the 
     * value of a <code>KeyboardEvent</code> created using the 
     * <code>DocumentEvent.createEvent</code> method. This method may only 
     * be called before the <code>KeyboardEvent</code> has been dispatched 
     * via the <code>EventTarget.dispatchEvent</code> method, though it may 
     * be called multiple times during that phase if necessary. If called 
     * multiple times, the final invocation takes precedence. This method 
     * has no effect if called after the event has been dispatched.
     * @param namespaceURI  Specifies the namespace URI associated with this 
     *   event, or <code>null</code> if the applications wish to have no 
     *   namespace. 
     * @param type  Specifies the event type. 
     * @param canBubbleArg Specifies whether or not the event can bubble.
     * @param cancelableArg Specifies whether or not the event's default 
     *   action can be prevent.
     * @param viewArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>AbstractView</code>.
     * @param keyIdentifierArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>keyIdentifier</code> attribute
     * @param keyLocationArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>keyLocation</code> attribute.
     * @param ctrlKeyArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>ctrlKey</code> attribute.
     * @param shiftKeyArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>shiftKey</code> attribute.
     * @param altKeyArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>altKey</code> attribute.
     * @param metaKeyArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>metaKey</code> attribute.
     * @param altGraphKeyArg Specifies the <code>KeyboardEvent</code>'s 
     *   <code>altGraphKey</code> attribute.
     */
    public void initKeyboardEventNS(String namespaceURI, 
                                    String type, 
                                    boolean canBubbleArg, 
                                    boolean cancelableArg, 
                                    AbstractView viewArg, 
                                    String keyIdentifierArg, 
                                    int keyLocationArg, 
                                    boolean ctrlKeyArg, 
                                    boolean shiftKeyArg, 
                                    boolean altKeyArg, 
                                    boolean metaKeyArg, 
                                    boolean altGraphKeyArg);

}
