/*
 * Copyright (c) 2001 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 */

package org.w3c.dom.xpath;

import org.w3c.dom.Node;
import org.w3c.dom.DOMException;

/**
 * The <code>XPathResult</code> interface represents the result of the 
 * evaluation of an XPath expression within the context of a particular 
 * node. Since evaluation of an XPath expression can result in various 
 * result types, this object makes it possible to discover and manipulate 
 * the type and value of the result.
 * <p>See also the <a href='http://www.w3.org/TR/2001/WD-DOM-Level-3-XPath-20010830'>Document Object Model (DOM) Level 3 XPath Specification</a>.
 */
public interface XPathResult {
    // XPathResultType
    /**
     * This code does not represent a specific type. An evaluation of an XPath 
     * expression will never produce this type. If this type is requested, 
     * then the evaluation must return whatever type naturally results from 
     * evaluation of the expression.
     */
    public static final short ANY_TYPE                  = 0;
    /**
     * The result is a number as defined by XPath 1.0.
     */
    public static final short NUMBER_TYPE               = 1;
    /**
     * The result is a string as defined by XPath 1.0.
     */
    public static final short STRING_TYPE               = 2;
    /**
     * The result is a boolean as defined by XPath 1.0.
     */
    public static final short BOOLEAN_TYPE              = 3;
    /**
     * The result is a node set as defined by XPath 1.0.
     */
    public static final short NODE_SET_TYPE             = 4;
    /**
     * The result is a single node, which may be any node of the node set 
     * defined by XPath 1.0, or null if the node set is empty. This is a 
     * convenience that permits optimization where the caller knows that no 
     * more than one such node exists because evaluation can stop after 
     * finding the one node of an expression that would otherwise return a 
     * node set (of type <code>NODE_SET_TYPE</code>).
     * <br>Where it is possible that multiple nodes may exist and the first 
     * node in document order is required, a <code>NODE_SET_TYPE</code> 
     * should be processed using an ordered iterator, because there is no 
     * order guarantee for a single node.
     */
    public static final short SINGLE_NODE_TYPE          = 5;

    /**
     * A code representing the type of this result, as defined by the type 
     * constants.
     */
    public short getResultType();

    /**
     * The value of this number result.
     * @exception XPathException
     *   TYPE_ERR: raised if <code>resultType</code> is not 
     *   <code>NUMBER_TYPE</code>.
     */
    public double getNumberValue()
                                           throws XPathException;

    /**
     * The value of this string result.
     * @exception XPathException
     *   TYPE_ERR: raised if <code>resultType</code> is not 
     *   <code>STRING_TYPE</code>.
     */
    public String getStringValue()
                                           throws XPathException;

    /**
     * The value of this boolean result.
     * @exception XPathException
     *   TYPE_ERR: raised if <code>resultType</code> is not 
     *   <code>BOOLEAN_TYPE</code>.
     */
    public boolean getBooleanValue()
                                           throws XPathException;

    /**
     * The value of this single node result, which may be null. This result is 
     * not guaranteed to be the first node in document order where the 
     * expression evaluates to multiple nodes.
     * @exception XPathException
     *   TYPE_ERR: raised if <code>resultType</code> is not 
     *   <code>SINGLE_NODE_TYPE</code>.
     */
    public Node getSingleNodeValue()
                                           throws XPathException;

    /**
     * Creates an <code>XPathSetIterator</code> which may be used to iterate 
     * over the nodes of the set of this result.
     * @param ordered The set must be iterated in document order.
     * @return An <code>XPathSetIterator</code> which may be used to iterate 
     *   the node set.
     * @exception XPathException
     *   TYPE_ERR: raised if <code>resultType</code> is not 
     *   <code>NODE_SET_TYPE</code>.
     * @exception DOMException
     *   INVALID_STATE_ERR: The document has been mutated since the result was 
     *   returned.
     */
    public XPathSetIterator getSetIterator(boolean ordered)
                                           throws XPathException, DOMException;

    /**
     * Creates an <code>XPathSetSnapshot</code> which lists the nodes of the 
     * set of this result. Unlike an iterator, after the snapshot has been 
     * requested, document mutation does not invalidate it.
     * @param ordered The set must be listed in document order.
     * @return An <code>XPathSetSnapshot</code> which may be used to list the 
     *   node set.
     * @exception XPathException
     *   TYPE_ERR: raised if <code>resultType</code> is not NODE_SET_TYPE.
     * @exception DOMException
     *   INVALID_STATE_ERR: The document has been mutated since the result was 
     *   returned.
     */
    public XPathSetSnapshot getSetSnapshot(boolean ordered)
                                           throws XPathException, DOMException;

}
