/*
 * Copyright (c) 2001 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 */

package org.w3c.dom.xpath;

import org.w3c.dom.Node;

/**
 *  The evaluation of XPath expressions is performed by 
 * <code>XPathEvaluator</code>, which will provide evaluation of XPath 1.0 
 * expressions with no specialized extension functions or variables. It is 
 * expected that the <code>XPathEvaluator</code> interface will be 
 * implemented on the same object which implements the <code>Document</code> 
 * interface in an implementation which supports the XPath DOM module. 
 * <code>XPathEvaluator</code> implementations may be available from other 
 * sources that may provide support for new versions of XPath or special 
 * extension functions or variables which are not defined in this 
 * specification. 
 * <p>See also the <a href='http://www.w3.org/TR/2001/WD-DOM-Level-3-XPath-20010830'>Document Object Model (DOM) Level 3 XPath Specification</a>.
 */
public interface XPathEvaluator {
    /**
     * Creates a parsed XPath expression with resolved namespaces. This is 
     * useful when an expression will be reused in an application since it 
     * makes it possible to compile the expression string into a more 
     * efficient internal form and preresolve all namespace prefixes which 
     * occur within the expression.
     * @param expression The XPath expression string to be parsed.
     * @param resolver The <code>resolver</code> permits translation of 
     *   prefixes within the XPath expression into appropriate namespace URIs
     *   . If this is specified as null, any namespace prefix within the 
     *   expression will result in <code>XPathException</code> being thrown 
     *   with the code <code>INVALID_EXPRESSION_ERR</code>.
     * @return The compiled form of the XPath expression.
     * @exception XPathException
     *   INVALID_EXPRESSION_ERR: Raised if the expression is not legal 
     *   according to the rules of the <code>XPathEvaluator</code> and the 
     *   in-scope namespace prefixes.
     */
    public XPathExpression createExpression(String expression, 
                                            XPathNSResolver resolver)
                                            throws XPathException;

    /**
     * Creates an XPath result object which may be passed as a parameter to 
     * the evaluation methods of this <code>XPathEvaluator</code> so that a 
     * new one is not created on each call to an evaluation method.
     * @return An empty <code>XPathEvaluator</code> with type 
     *   <code>ANY_TYPE</code>.
     */
    public XPathResult createResult();

    /**
     * Adapts any DOM node to resolve namespaces so that an XPath expression 
     * can be easily evaluated relative to the context of the node where it 
     * appeared within the document.
     * @param nodeResolver The node to be used as a context for namespace 
     *   resolution.
     * @return <code>XPathNSResolver</code> which resolves namespaces with 
     *   respect to the definitions in scope for a specified node.
     */
    public XPathNSResolver createNSResolver(Node nodeResolver);

    /**
     * Evaluates an XPath expression string and returns a result of the 
     * specified type if possible.
     * @param expression The XPath expression string to be parsed and 
     *   evaluated.
     * @param contextNode The <code>context</code> is context node for the 
     *   evaluation of this XPath expression.
     * @param resolver The <code>resolver</code> permits translation of 
     *   prefixes within the XPath expression into appropriate namespace URIs
     *   . If this is specified as null, any namespace prefix within the 
     *   expression will result in <code>XPathException</code> being thrown 
     *   with the code <code>INVALID_EXPRESSION_ERR</code>.
     * @param type If a specific <code>type</code> is specified, then the 
     *   result will be coerced to return the specified type relying on 
     *   XPath conversions and fail if the desired coercion is not possible. 
     *   This must be one of the type codes of <code>XPathResult</code>.
     * @param result The <code>result</code> specifies a specific 
     *   <code>XPathResult</code> to be reused and returned by this method. 
     *   If this is specified as null, a new <code>XPathResult</code> will 
     *   be constructed and returned. Any <code>XPathResult</code> which was 
     *   not created by this <code>XPathEvaluator</code> may be ignored as 
     *   though a null were passed as the parameter.
     * @return The result of the evaluation of the XPath expression.
     * @exception XPathException
     *   INVALID_EXPRESSION_ERR: Raised if the expression is not legal 
     *   according to the rules of the <code>XPathEvaluator</code> and the 
     *   in-scope namespace prefixes.
     *   <br>TYPE_ERR: Raised if the result cannot be converted to return the 
     *   specified type.
     */
    public XPathResult evaluate(String expression, 
                                Node contextNode, 
                                XPathNSResolver resolver, 
                                short type, 
                                XPathResult result)
                                throws XPathException;

    /**
     * Evaluates an XPath expression and returns a result.
     * @param expression The XPath expression to be evaluated.
     * @param contextNode The <code>context</code> is context node for the 
     *   evaluation of this XPath expression.
     * @param type If a specific <code>type</code> is specified, then the 
     *   result will be coerced to return the specified type relying on 
     *   XPath conversions and fail if the desired coercion is not possible. 
     *   This must be one of the type codes of <code>XPathResult</code>.
     * @param result The <code>result</code> specifies a specific 
     *   <code>XPathResult</code> to be reused and returned by this method. 
     *   If this is specified as null, a new <code>XPathResult</code> will 
     *   be constructed and returned. Any <code>XPathResult</code> which was 
     *   not created by this <code>XPathEvaluator</code> may be ignored as 
     *   though a null were passed as the parameter.
     * @return The result of the evaluation of the XPath expression.
     * @exception XPathException
     *   TYPE_ERR: Raised if the result cannot be converted to return the 
     *   specified type.
     */
    public XPathResult evaluateExpression(XPathExpression expression, 
                                          Node contextNode, 
                                          short type, 
                                          XPathResult result)
                                          throws XPathException;

}
