/*
 * Copyright (c) 2001 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 */

package org.w3c.dom.xpath;

import org.w3c.dom.Node;

/**
 *  The evaluation of XPath expressions is available in 
 * <code>XPathEvaluator</code>. A <code>Document</code> which implements the 
 * XPath module will be castable using language-specific mechanisms to 
 * <code>XPathEvaluator</code>, which will provide evaluation of XPath 
 * expressions with no special extension functions or variables. 
 * <code>XPathEvaluator</code> implementations may be available from other 
 * sources that provide extension functions or variables.  should we return 
 * an Object as the result instead and have one general 
 * <code>evaluateAs</code> method? 
 * <p>See also the <a href='http://www.w3.org/TR/2001/WD-DOM-Level-3-XPath-20010618'>Document Object Model (DOM) Level 3 XPath Specification</a>.
 */
public interface XPathEvaluator {
    /**
     * Evaluates an XPath expression and converts the result to a boolean.
     * @param expression The XPath expression to be evaluated.
     * @param contextNode The context node for the evaluation of the XPath 
     *   expression.
     * @param resolver The <code>resolver</code> permits translation of 
     *   prefixes within the XPath expression into appropriate namespaceURIs.
     * @return The result of the evaluation of the XPath expression.
     * @exception XPathException
     *   TYPE_ERR: Raised if the result cannot be converted to a boolean
     */
    public boolean evaluateAsBoolean(String expression, 
                                     Node contextNode, 
                                     NamespaceResolver resolver)
                                     throws XPathException;

    /**
     *  Evaluates an XPath expression and converts the result to a number.
     * @param expression The XPath expression to be evaluated.
     * @param contextNode The context node for the evaluation of the XPath 
     *   expression.
     * @param resolver The <code>resolver</code> permits translation of 
     *   prefixes within the XPath expression into appropriate namespaceURIs.
     * @return The result of the evaluation of the XPath expression.
     * @exception XPathException
     *   TYPE_ERR: Raised if the result cannot be converted to a number (IEEE 
     *   double precision floating point value)
     */
    public double evaluateAsNumber(String expression, 
                                   Node contextNode, 
                                   NamespaceResolver resolver)
                                   throws XPathException;

    /**
     *  Evaluates an XPath expression and converts the result to a string.
     * @param expression The XPath expression to be evaluated.
     * @param contextNode The context node for the evaluation of the XPath 
     *   expression.
     * @param resolver The <code>resolver</code> permits translation of 
     *   prefixes within the XPath expression into appropriate namespaceURIs.
     * @return The result of the evaluation of the XPath expression.
     * @exception XPathException
     *   TYPE_ERR: Raised if the result cannot be converted to a 
     *   <code>DOMString</code>
     */
    public String evaluateAsString(String expression, 
                                   Node contextNode, 
                                   NamespaceResolver resolver)
                                   throws XPathException;

    /**
     * Evaluates an XPath expression and returns the first node of the 
     * resulting set, <code>null</code> if the resulting set is empty.
     * @param expression The XPath expression to be evaluated.
     * @param contextNode The context node for the evaluation of the XPath 
     *   expression.
     * @param resolver The <code>resolver</code> permits translation of 
     *   prefixes within the XPath expression into appropriate namespaceURIs.
     * @return The result of the evaluation of the XPath expression.
     * @exception XPathException
     *   TYPE_ERR: Raised if the result cannot be converted to a node set.
     */
    public Node evaluateAsNode(String expression, 
                               Node contextNode, 
                               NamespaceResolver resolver)
                               throws XPathException;

    /**
     * Evaluates an XPath expression and returns the result as a node set.
     * @param expression The XPath expression to be evaluated.
     * @param contextNode The context node for the evaluation of the XPath 
     *   expression.
     * @param resolver The <code>resolver</code> permits translation of 
     *   prefixes within the XPath expression into appropriate namespaceURIs.
     * @return The result of the evaluation of the XPath expression.
     * @exception XPathException
     *   TYPE_ERR: Raised if the result cannot be converted to a node set.
     */
    public ActiveNodeSet evaluateAsNodeSet(String expression, 
                                           Node contextNode, 
                                           NamespaceResolver resolver)
                                           throws XPathException;

}
