/*
 * Copyright (c) 2001 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 */

package org.w3c.dom.xpath;

import org.w3c.dom.Node;
import org.w3c.dom.DOMException;

/**
 * This represents a live node set obtained with the evaluation of an XPath 
 * expression.Do we need a detach() method?
 * <p>See also the <a href='http://www.w3.org/TR/2001/WD-DOM-Level-3-XPath-20010618'>Document Object Model (DOM) Level 3 XPath Specification</a>.
 */
public interface ActiveNodeSet {
    /**
     * Returns the next node from the XPath expression evaluation.
     * @return Returns the next node.
     * @exception DOMException
     *   INVALID_STATE_ERR: The <code>ActiveNodeSet</code> is no longer valid.
     */
    public Node nextNode()
                         throws DOMException;

    /**
     * @exception DOMException
     *   INVALID_STATE_ERR: The <code>ActiveNodeSet</code> is no longer valid.
     */
    public void reset()
                      throws DOMException;

    /**
     * Clones the <code>ActiveNodeSet</code>. It was suggested that cloneSet 
     * should copy only the remaining set, which might favor incremental 
     * implementations and also make it possible to non-destructively pass 
     * on the current position. Clones must then be constructed at the 
     * appropriate position within the set and complete clones may only be 
     * constructed before processing any nodes. 
     * @return The new <code>ActiveNodeSet</code>.
     * @exception DOMException
     *   INVALID_STATE_ERR: The <code>ActiveNodeSet</code> is no longer valid.
     */
    public ActiveNodeSet cloneSet()
                                  throws DOMException;

    /**
     * This method may be called as long as the set is not invalid to get a 
     * set that is sorted into document order. Nodes which have no defined 
     * order with respect to each other (such as <code>Attr</code> nodes 
     * attached to an <code>Element</code>) will not be ordered in any 
     * particular order with respect to each other by this method. Here are 
     * significant differences between DOM's currently proposed document 
     * order and the XPath model's document order expected by XSL and other 
     * callers of <code>getDocumentOrderedSet</code>. Some of the 
     * differences, we could just adopt. Others are unclear.For example, in 
     * the fragment 
     * <code>&lt;foo bar1="abc"/&gt;&lt;foo bar2="def" bar3="ghi"&gt;</code>
     * , DOM currently says that attribute nodes bar1 and bar2 are unordered 
     * with respect to each other, since they each have no parent. XPath and 
     * DOM both agree that bar2 and bar3 have no order with respect to each 
     * other, but XPath gives them an order with respect to their parent, 
     * giving them a definate position in the resulting sorted node set, 
     * whereas DOM does not acknowledge the ownerElement as part of the 
     * ordering relationship and makes it impossible to meaningfully sort 
     * attributes nodes of an element with elements or the attributes of 
     * other elements. I think we should adjust DOM's proposed comparison 
     * APIs and the result will help, rather than hurting, the DOM 
     * specification.It will be much more difficult to make Attr nodes 
     * masquerading as XPath namespace nodes XPath-compatible with respect 
     * to their sort order. If it is a requirement to make this work, we 
     * have to use real Namespace nodes, IMO.
     * @return 
     * @exception DOMException
     *   INVALID_STATE_ERR: The <code>ActiveNodeSet</code> is no longer valid.
     */
    public ActiveNodeSet getDocumentOrderedSet()
                                               throws DOMException;

    /**
     * This method may be called as long as the set is not invalid to force 
     * complete evaluation and return <code>StaticNodeSet</code>, which 
     * never becomes invalid and never changes the array of node references 
     * even if the nodes are no longer are in the hierarchy or no longer 
     * match the expression that created the set.
     * @return The complete evaluation of the XPath expression.
     * @exception DOMException
     *   INVALID_STATE_ERR: The <code>ActiveNodeSet</code> is no longer valid.
     */
    public StaticNodeSet getStaticNodeSet()
                                          throws DOMException;

}
