/*
 * Copyright (c) 2001 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 */

package org.w3c.dom.loadSave;

import org.w3c.dom.Node;
import org.w3c.dom.DOMSystemException;

/**
 *  DOMWriter provides an API for serializing (writing) a DOM document out in 
 * the form of a source XML document. The XML data is written to an output 
 * stream, the type of which depends on the specific language bindings in 
 * use. 
 * <p>Three options are available for the general appearance of the formatted 
 * output: As-is, canonical and reformatted. As-is formatting leaves all 
 * "white space in element content" and new-lines unchanged. If the DOM 
 * document originated as XML source, and if all white space was retained, 
 * this option will come the closest to recovering the format of the 
 * original document. (There may still be differences due to normalization 
 * of attribute values and new-line characters or the handling of character 
 * references.)  Canonical formatting writes the document according to the 
 * rules specified by W3C Canonical XML Version 1.0.  
 * http://www.w3.org/TR/xml-c14n  Reformatted output has white space and 
 * newlines adjusted to produce a pretty-printed, indented, human-readable 
 * form. The exact form of the transformations is not specified. 
 * <p><code>DOMWriter</code> accepts any node type for serialization. For 
 * nodes of type <code>Document</code> or <code>Entity</code>, well formed 
 * XML will be created. The serialized output for these node types is either 
 * as a Document or an External Entity, respectively, and is acceptable 
 * input for an XML parser. For all other types of nodes the serialized form 
 * is not specified, but should be something useful to a human for debugging 
 * or diagnostic purposes. Note: rigorously designing an external (source) 
 * form for stand-alone node types that don't already have one defined by 
 * the XML rec seems a bit much to take on here.
 * <p>Within a Document or Entity being serialized, Nodes are processed as 
 * follows Documents are written including an XML declaration and a DTD 
 * subset, if one exists in the DOM. Writing a document node serializes the 
 * entire document.  Entity nodes, when written directly by 
 * <code>DOMWriter.writeNode()</code>, output a Text Decl and the entity 
 * expansion. The resulting output will be valid as an external entity.  No 
 * output is generated for any entity nodes when writing a 
 * <code>Document</code>.  Entity References nodes are serializes as an 
 * entity reference of the form <code>"&amp;entityName;"</code>) in the 
 * output. Child nodes (the expansion) of the entity reference are ignored.  
 * CDATA sections containing content characters that can not be represented 
 * in the specified output encoding are handled according to the 
 * "split-cdata-sections" option. If the option is true, CDATA sections are 
 * split, and the unrepresentable characters are serialized as numeric 
 * character references in ordinary content. The exact position and number 
 * of splits is not specified. If the option is false, unrepresentable 
 * characters in a CDATA section are reported as errors. The error is not 
 * recoverable - there is no mechanism for supplying alternative characters 
 * and continuing with the serialization. All other node types (Element, 
 * Text, etc.) are serialized to their corresponding XML source form. 
 * <p> Within the character data of a document (outside of markup), any 
 * characters that cannot be represented directly are replaced with 
 * character references. Occurrences of '&lt;' and '&amp;' are replaced by 
 * the predefined entities &amp;lt; and &amp;amp. The other predefined 
 * entities (&amp;gt, &amp;apos, etc.) are not used; these characters can be 
 * included directly. Any character that can not be represented directly in 
 * the output character encoding is serialized as a numeric character 
 * reference. 
 * <p> Attributes not containing quotes are serialized in quotes. Attributes 
 * containing quotes but no apostrophes are serialized in apostrophes 
 * (single quotes). Attributes containing both forms of quotes are 
 * serialized in quotes, with quotes within the value represented by the 
 * predefined entity &amp;quot;. Any character that can not be represented 
 * directly in the output character encoding is serialized as a numeric 
 * character reference. 
 * <p> Within markup, but outside of attributes, any occurrence of a character 
 * that cannot be represented in the output character encoding is reported 
 * as an error. An example would be serializing the element 
 * &lt;LaCaada/&gt; with the encoding=US-ASCII 
 * <p> Unicode Character Normalization. When requested by setting the 
 * <code>normalizeCharacters</code> option on DOMWriter, all data to be 
 * serialized, both markup and character data, is normalized according to 
 * the rules defined by Unicode Canonical Composition, Normalization Form C. 
 * The normalization process affects only the data as it is being written; 
 * it does not alter the DOM's view of the document after serialization has 
 * completed. The W3C character model and normalization are described at  
 * http://www.w3.org/TR/charmod/#TextNormalization. Unicode normalization 
 * forms are described at  http://www.unicode.org/unicode/reports/tr15/ 
 * <p>Name space checking and fixup during serialization is a user option. 
 * When the option is selected, the serialization process will verify that 
 * name space declarations, name space prefixes and the name space URIs 
 * associated with Elements and Attributes are consistent. If 
 * inconsistencies are found, the serialized form of the document will be 
 * altered to remove them. The exact form of the alterations are not 
 * defined, and are implementation dependent. 
 * <p>Any changes made affect only the name space prefixes and declarations 
 * appearing in the serialized data. The DOM's view of the document is not 
 * altered by the serialization operation, and does not reflect any changes 
 * made to name space declarations or prefixes in the serialized output. 
 * <p><code>DOMWriter</code>s have a number of named properties that can be 
 * queried or set. Here is a list of properties that must be recognized by 
 * all implementations. normalizeCharacters true: Perform Unicode 
 * Normalization of the characters in document as they are written out. Only 
 * the characters being written are (potentially) altered. The DOM document 
 * itself is unchanged.false: do not perform character normalization.
 * default: true.supported values: true: required; false: required.
 * namespaceFixuptrue: Check namespace declarations and prefixes for 
 * consistency, and fix them in the serialized data if they are 
 * inconsistent. false: Perform no special checks on name space 
 * declarations, prefixes or URIs.default: true;supported values: true: 
 * required; false: required.split-cdata-sectionstrue: Split CDATA sections 
 * containing characters that can not be represented in the output encoding, 
 * and output the characters using numeric character references.false: 
 * Signal an error if a CDATA section contains an unrepresentable character.
 * supported values: true: required; false: required.
 * <p>See also the <a href='http://www.w3.org/TR/2001/WD-DOM-Level-3-CMLS-20010419'>Document Object Model (DOM) Level 3 Content Models and Load
and Save Specification</a>.
 */
public interface DOMWriter {
    /**
     *  The character encoding in which the output will be written. 
     * <br> The encoding to use when writing is determined as follows: If the 
     * encoding attribute has been set, that value will be used.If the 
     * encoding attribute is null or empty, but the item to be written 
     * includes an encoding declaration, that value will be used.If neither 
     * of the above provides an encoding name, a default encoding of "utf-8" 
     * will be used.
     * <br>The default value is null.
     */
    public String getEncoding();
    /**
     *  The character encoding in which the output will be written. 
     * <br> The encoding to use when writing is determined as follows: If the 
     * encoding attribute has been set, that value will be used.If the 
     * encoding attribute is null or empty, but the item to be written 
     * includes an encoding declaration, that value will be used.If neither 
     * of the above provides an encoding name, a default encoding of "utf-8" 
     * will be used.
     * <br>The default value is null.
     */
    public void setEncoding(String encoding);

    /**
     *  The actual character encoding that was last used by this formatter. 
     * This convenience method allows the encoding that was used when 
     * serializing a document to be directly obtained. 
     */
    public String getLastEncoding();

    /**
     *  As-is, canonical or reformatted. Need to add constants for these. 
     * <br> The default value is as-is. 
     */
    public short getFormat();
    /**
     *  As-is, canonical or reformatted. Need to add constants for these. 
     * <br> The default value is as-is. 
     */
    public void setFormat(short format);

    /**
     * The end-of-line character(s) to be used in the XML being written out. 
     * The only permitted values are these: null: Use a default end-of-line 
     * sequence. DOM implementations should choose the default to match the 
     * usual convention for text files in the environment being used. 
     * Implementations must choose a default sequence that matches one of 
     * those allowed by the XML Recommendation, 
     * http://www.w3.org/TR/REC-xml#sec-line-endsCRCR-LFLF
     * <br>The default value for this attribute is null.
     * @version DOM Level 3
     */
    public String getNewLine();
    /**
     * The end-of-line character(s) to be used in the XML being written out. 
     * The only permitted values are these: null: Use a default end-of-line 
     * sequence. DOM implementations should choose the default to match the 
     * usual convention for text files in the environment being used. 
     * Implementations must choose a default sequence that matches one of 
     * those allowed by the XML Recommendation, 
     * http://www.w3.org/TR/REC-xml#sec-line-endsCRCR-LFLF
     * <br>The default value for this attribute is null.
     * @version DOM Level 3
     */
    public void setNewLine(String newLine);

    /**
     * Write out the specified node as described above in the description of 
     * <code>DOMWriter</code>. Writing a Document or Entity node produces a 
     * serialized form that is well formed XML. Writing other node types 
     * produces a fragment of text in a form that is not fully defined by 
     * this document, but that should be useful to a human for debugging or 
     * diagnostic purposes. 
     * @param destination The destination for the data to be written.
     * @param node The <code>Document</code> or <code>Entity</code> node to 
     *   be written. For other node types, something sensible should be 
     *   written, but the exact serialized form is not specified.
     * @exception DOMSystemException
     *   This exception will be raised in response to any sort of IO or system 
     *   error that occurs while writing to the destination. It may wrap an 
     *   underlying system exception.
     */
    public void writeNode(java.io.OutputStream destination, 
                          Node node)
                          throws DOMSystemException;

}
