/*
 * Copyright (c) 2001 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 */

package org.w3c.dom.loadSave;

import org.w3c.dom.Element;

/**
 * <code>DOMBuilderFilter</code>s provide applications the ability to examine 
 * Element nodes as they are being constructed during a parse. As each 
 * elements is examined, it may be modified or removed, or the entire parse 
 * may be terminated early. 
 * <p>At the time any of the filter methods are called by the parser, the 
 * owner Document and DOMImplementation objects exist and are accessible.
 * <p>All validity checking while reading a document occurs on the source 
 * document as it appears on the input stream, not on the DOM document as it 
 * is built in memory. With filters, the document in memory may be a subset 
 * of the document on the stream, and its validity may have been affected by 
 * the filtering.
 * <p>See also the <a href='http://www.w3.org/TR/2001/WD-DOM-Level-3-CMLS-20010419'>Document Object Model (DOM) Level 3 Content Models and Load
and Save Specification</a>.
 */
public interface DOMBuilderFilter {
    /**
     * This method will be called by the parser after each Element start tag 
     * has been scanned, but before the remainder of the Element is 
     * processed. The intent is to allow the element, including any 
     * children, to be efficiently skipped.
     * <br>The element node passed to <code>startElement</code> for filtering 
     * will include all of the Element's attributes, but none of the 
     * children nodes. The Element may not yet be in place in the document 
     * being constructed (it may not have a parent node.) 
     * <br>A StartElement filter function may access or change the attributers 
     * for the Element. Changing NameSpace declarations will have no effect 
     * on name space resolution by the parser.
     * <br>For efficiency, the Element node passed to the filter may not be 
     * the same one as is actually placed in the tree if the node is 
     * accepted. And the actual node (node object identity) may be reused 
     * during the process of reading in and filtering a document.
     * @param element The newly encountered element. At the time this method 
     *   is called, the element is incomplete - it will have its attributes, 
     *   but no children. 
     * @return return true if this Element should be included in the DOM 
     *   document being built. Return false if the Element and all of its 
     *   children should be skipped.
     */
    public boolean startElement(Element element);

    /**
     * This method will be called by the parser at the completion of the parse 
     * of each element. The element node will exist and be complete, as will 
     * all of its children, and their children, recursively. The element's 
     * parent node will also exist, although that node may be incomplete, as 
     * it may have additional children that have not yet been parsed.
     * <br>From within this method, the new node may be freely modified - 
     * children may be added or removed, text nodes modified, etc. This node 
     * may also be removed from its parent node, which will prevent it from 
     * appearing in the final document at the completion of the parse. Aside 
     * from this one operation on the node's parent, the state of the rest 
     * of the document outside of this node is not defined, and the affect 
     * of any attempt to navigate to or modify any other part of the 
     * document is undefined. 
     * <br>For validating parsers, the checks are made on the original 
     * document, before any modification by the filter. No validity checks 
     * are made on any document modifications made by the filter. 
     * @param element The newly constructed element. At the time this method 
     *   is called, the element is complete - it has all of its children 
     *   (and their children, recursively) and attributes, and is attached 
     *   as a child to its parent. 
     * @return return true 
     */
    public boolean endElement(Element element);

}
