/*
 * Copyright (c) 2001 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 */

package org.w3c.dom.loadSave;

import org.w3c.dom.DOMErrorHandler;
import org.w3c.dom.Document;
import org.w3c.dom.DOMSystemException;
import org.w3c.dom.DOMException;

/**
 * A parser interface.
 * <p><code>DOMBuilder</code> provides an API for parsing XML documents and 
 * building the corresponding DOM document tree. A <code>DOMBuilder</code> 
 * instance is obtained from the <code>DOMImplementationLS</code> interface 
 * by invoking its <code>createDOMBuilder()</code>method.
 * <p><code>DOMBuilder</code>s have a number of named properties that can be 
 * queried or set. Here is a list of properties that must be recognized by 
 * all implementations. namespaces true: perform Namespace processing.false: 
 * do not perform name space processing.default: true.supported values: 
 * true: required; false: optionalnamespace-declarationstrue: include 
 * namespace declarations (xmlns attributes) in the DOM document. false: 
 * discard all namespace declarations. In either case, namespace prefixes 
 * will be retained. default: true. supported values: true: required; false: 
 * optionalvalidation true: report validation errors (setting true also will 
 * force the external-general-entities and external-parameter-entities 
 * properties to be set true.) Also note that the <code>validate-if-cm</code>
 *  feature will alter the validation behavior when this feature is set true.
 * false: do not report validation errors.default: false.supported values: 
 * true: optional; false: requiredexternal-general-entitiestrue: include all 
 * external general (text) entities.false: do not include external general 
 * entities.default: true.supported values: true: required; false: optional
 * external-parameter-entitiestrue: include all external parameter entities.
 * false: do not include external parameter entities.default: true.supported 
 * values: true: required; false: optionalvalidate-if-cm true: when both 
 * this feature and validation are true, enable validation only when the 
 * document being processed has a content model. Documents without content 
 * models are parsed without validation.false: the validation feature alone 
 * controls whether the document is checked for validity. Documents without 
 * content models are not valid.default: false.supported values: true: 
 * optional; false: requiredcreate-entity-ref-nodestrue: create entity 
 * reference nodes in the DOM document. Setting this value true will also 
 * set create-entity-nodes to be truefalse: omit all entity reference nodes 
 * from the DOM document, putting the entity expansions directly in their 
 * place.default: true.supported values: true: required; false: optional
 * entity-nodestrue: create entity nodes in the DOM document.false: omit all 
 * entity nodes from the DOM document. Setting this value false will also 
 * set create-entity-ref-nodes false.default: true.supported values: true: 
 * required; false: optionalwhite-space-in-element-contenttrue: include 
 * white space in element content in the DOM document. This is sometimes 
 * referred to as ignorable white spacefalse: omit said white space. Note 
 * that white space in element content will only be omitted if it can be 
 * identified as such, and not all parsers may be able to do so. default: 
 * true.supported values: true: required; false: optionalcdata-nodestrue: 
 * Create DOM CDATA nodes in response to the appearance of CDATA sections in 
 * the source XML. false: Do not create CDATA nodes in the DOM document. The 
 * content of any CDATA sections in the source XML appears in the DOM as if 
 * it had been normal (non-CDATA) content. If a CDATA section is adjacent to 
 * other content, the combined content appears in a single TEXT node. The 
 * DOM Document produced by the DOMBuilder will not have adjacent TEXT nodes.
 * default: truesupported values: false: optional; true: requiredcomments
 * true: Include XML comments in the DOM documentfalse: Discard XML 
 * comments, do not create Comment nodes in the DOM Document resulting from 
 * a parse. default: truesupported values: false: required; true: required
 * charset-overrides-xml-encodingtrue: If a higher level protocol such as 
 * http provides an indication of the character encoding of the input stream 
 * being processed, that will override any encoding specified in the XML or 
 * TEXT declaration of the XML. Explicitly setting an encoding in the 
 * DOMInputSource overrides encodings from the protocol. false: Any 
 * character set encoding information from higher level protocols is ignored 
 * by the parser. default: truesupported values: false: required; true: 
 * required
 * <p>See also the <a href='http://www.w3.org/TR/2001/WD-DOM-Level-3-CMLS-20010419'>Document Object Model (DOM) Level 3 Content Models and Load
and Save Specification</a>.
 */
public interface DOMBuilder {
    /**
     * If a <code>DOMEntityResolver</code> has been specified, each time a 
     * reference to an external entity is encountered the 
     * <code>DOMBuilder</code> will pass the public and system IDs to the 
     * entity resolver, which can then specify the actual source of the 
     * entity.
     */
    public DOMEntityResolver getEntityResolver();
    /**
     * If a <code>DOMEntityResolver</code> has been specified, each time a 
     * reference to an external entity is encountered the 
     * <code>DOMBuilder</code> will pass the public and system IDs to the 
     * entity resolver, which can then specify the actual source of the 
     * entity.
     */
    public void setEntityResolver(DOMEntityResolver entityResolver);

    /**
     * In the event that an error is encountered in the XML document being 
     * parsed, the <code>DOMDcoumentBuilder</code> will call back to the 
     * <code>errorHandler</code> with the error information.The 
     * DOMErrorHandler interface is being developed separately, in 
     * conjunction with the design of the content model and validation 
     * module.
     */
    public DOMErrorHandler getErrorHandler();
    /**
     * In the event that an error is encountered in the XML document being 
     * parsed, the <code>DOMDcoumentBuilder</code> will call back to the 
     * <code>errorHandler</code> with the error information.The 
     * DOMErrorHandler interface is being developed separately, in 
     * conjunction with the design of the content model and validation 
     * module.
     */
    public void setErrorHandler(DOMErrorHandler errorHandler);

    /**
     * When the application provides a filter, the parser will call out to the 
     * filter at the completion of the construction of each 
     * <code>Element</code> node. The filter implementation can choose to 
     * remove the element from the document being constructed or to 
     * terminate the parse early.
     */
    public DOMBuilderFilter getFilter();
    /**
     * When the application provides a filter, the parser will call out to the 
     * filter at the completion of the construction of each 
     * <code>Element</code> node. The filter implementation can choose to 
     * remove the element from the document being constructed or to 
     * terminate the parse early.
     */
    public void setFilter(DOMBuilderFilter filter);

    /**
     * Set the state of a feature.
     * <br>The feature name has the same form as a DOM hasFeature string.
     * <br>It is possible for a <code>DOMBuilder</code> to recognize a feature 
     * name but to be unable to set its value.
     * @param name The feature name, which is a DOM has-feature style string.
     * @param state The requested state of the feature (true or false).
     * @exception DOMException
     *   Raise a NOT_SUPPORTED_ERR exception When the <code>DOMBuilder</code> 
     *   recognizes the feature name but cannot set the requested value. 
     *   <br>Raise a NOT_FOUND_ERR When the <code>DOMBuilder</code> does not 
     *   recognize the feature name.
     */
    public void setFeature(String name, 
                           boolean state)
                           throws DOMException;

    /**
     * query whether the <code>DOMBuilder</code> recognizes a feature name.
     * <br>The feature name has the same form as a DOM hasFeature string.
     * <br>It is possible for a <code>DOMBuilder</code> to recognize a feature 
     * name but to be unable to set its value. For example, a non-validating 
     * parser would recognize the feature "validation", would report that 
     * its value was false, and would raise an exception if an attempt was 
     * made to enable validation by setting the feature to true. 
     * @param name The feature name, which has the same syntax as a DOM 
     *   has-feature string.
     * @return true if the feature name is recognized by the 
     *   <code>DOMBuilder</code>. False if the feature name is not 
     *   recognized.
     */
    public boolean supportsFeature(String name);

    /**
     * query whether setting a feature is supported.
     * <br>The feature name has the same form as a DOM hasFeature string.
     * <br>It is possible for a <code>DOMBuilder</code> to recognize a feature 
     * name but to be unable to set its value.
     * @param name The feature name, which is a DOM has-feature style string.
     * @param state The requested state of the feature (true or false).
     * @return true if the feature could be successfully set to the specified 
     *   value, or false if the feature is not recognized or the requested 
     *   value is not supported. The value of the feature itself is not 
     *   changed.
     */
    public boolean canSetFeature(String name, 
                                 boolean state);

    /**
     * Look up the value of a feature.
     * <br>The feature name has the same form as a DOM hasFeature string
     * @param name The feature name, which is a string with DOM has-feature 
     *   syntax.
     * @return The current state of the feature (true or false).
     * @exception DOMException
     *   Raise a NOT_FOUND_ERR When the <code>DOMBuilder</code> does not 
     *   recognize the feature name.
     */
    public boolean getFeature(String name)
                              throws DOMException;

    /**
     * Parse an XML document from a location identified by an URI.
     * @param uri The location of the XML document to be read. 
     * @return The newly created and populated<code>Document</code>.
     * @exception DOMException
     *   Exceptions raised by <code>parseURI()</code> originate with the 
     *   installed ErrorHandler, and thus depend on the implementation of 
     *   the <code>DOMErrorHandler</code> interfaces. The default error 
     *   handlers will raise a <code>DOMException</code> if any form of XML 
     *   validation or well formedness error or warning occurs during the 
     *   parse, but application defined errorHandlers are not required to do 
     *   so. 
     * @exception DOMSystemException
     *   Exceptions raised by <code>parseURI()</code> originate with the 
     *   installed ErrorHandler, and thus depend on the implementation of 
     *   the <code>DOMErrorHandler</code> interfaces. The default error 
     *   handlers will raise a DOMSystemException if any form I/O or other 
     *   system error occurs during the parse, but application defined error 
     *   handlers are not required to do so. 
     */
    public Document parseURI(String uri)
                             throws DOMException, DOMSystemException;

    /**
     * Parse an XML document from a location identified by an 
     * <code>DOMInputSource</code>.
     * @param is The <code>DOMInputSource</code> from which the source 
     *   document is to be read. 
     * @return The newly created and populated<code>Document</code>.
     * @exception DOMException
     *   Exceptions raised by <code>parseDOMInputSource()</code> originate 
     *   with the installed ErrorHandler, and thus depend on the 
     *   implementation of the <code>DOMErrorHandler</code> interfaces. The 
     *   default ErrorHandlers will raise a <code>DOMException</code> if any 
     *   form of XML validation or well formedness error or warning occurs 
     *   during the parse, but application defined errorHandlers are not 
     *   required to do so. 
     * @exception DOMSystemException
     *   Exceptions raised by <code>parseDOMInputSource()</code> originate 
     *   with the installed ErrorHandler, and thus depend on the 
     *   implementation of the <code>DOMErrorHandler</code> interfaces. The 
     *   default ErrorHandlers will raise a <code>DOMSystemException</code> 
     *   if any form I/O or other system error occurs during the parse, but 
     *   application defined ErrorHandlers are not required to do so. 
     */
    public Document parseDOMInputSource(DOMInputSource is)
                                        throws DOMException, DOMSystemException;

}
