/*
 * Copyright (c) 2001 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de
 * Recherche en Informatique et en Automatique, Keio University). All
 * Rights Reserved. This program is distributed under the W3C's Software
 * Intellectual Property License. This program is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.
 * See W3C License http://www.w3.org/Consortium/Legal/ for more details.
 */

package org.w3c.dom.contentModel;

import org.w3c.dom.nsElement;
import org.w3c.dom.DOMException;

/**
 * <code>CMModel</code> is an abstract object that could map to a DTD, an XML 
 * Schema, a database schema, etc. It's a generalized content model object, 
 * that has both an internal and external subset. The internal subset would 
 * always exist, even if empty, with the external subset (if present) being 
 * represented as by an "active" <code>CMExternalModel</code>. Many 
 * <code>CMExternalModel</code>s could exist, but only one can be specified 
 * as "active"; it is also possible that none are "active". The issue of 
 * multiple content models is misleading since in this architecture, only 
 * one <code>CMModel</code> exists, with an internal subset that references 
 * the external subset. If the external subset changes to another "acitve" 
 * <code>CMExternalModel</code>, the internal subset is "fixed up." The 
 * CMModel also contains the factory methods needed to create a various 
 * types of CMNodes like <code>CMElementDeclaration</code>, 
 * <code>CMAttributeDeclaration</code>, etc. 
 * <p>See also the <a href='http://www.w3.org/TR/2001/WD-DOM-Level-3-CMLS-20010419'>Document Object Model (DOM) Level 3 Content Models and Load
and Save Specification</a>.
 */
public interface CMModel extends CMNode {
    /**
     * True if this content model defines the document structure in terms of 
     * namespaces and local names; false if the document structure is 
     * defined only in terms of <code>QNames</code>.
     */
    public boolean getIsNamespaceAware();

    /**
     * The root element declaration for the content model. Although a root 
     * element is specified in the document instance, when a content model 
     * is generated, a user should be able to chose the root element for 
     * editing purpose. This is just a placeholder for that element. It 
     * could also be null. For validating an XML document, root element must 
     * be defined in its active content model. CMModel.rootElementDecl 
     * provides access to that root element declaration. This recommendation 
     * does not say how to fill in the rootElementdecl. It could be manually 
     * done by the user before validating a document, in some cases where 
     * possible, the CMModle loader may be able to fill it in etc.
     */
    public CMElementDeclaration getRootElementDecl();
    /**
     * The root element declaration for the content model. Although a root 
     * element is specified in the document instance, when a content model 
     * is generated, a user should be able to chose the root element for 
     * editing purpose. This is just a placeholder for that element. It 
     * could also be null. For validating an XML document, root element must 
     * be defined in its active content model. CMModel.rootElementDecl 
     * provides access to that root element declaration. This recommendation 
     * does not say how to fill in the rootElementdecl. It could be manually 
     * done by the user before validating a document, in some cases where 
     * possible, the CMModle loader may be able to fill it in etc.
     */
    public void setRootElementDecl(CMElementDeclaration rootElementDecl);

    /**
     * Location of the document describing the content model defined in this 
     * CMModel.
     * @return This method returns a DOMString defining the absolute location 
     *   from which this document is retrieved including the document name.
     */
    public String getLocation();

    /**
     * Determines namespace of <code>CMModel</code>.
     * @return Namespace of <code>CMModel</code>.
     */
    public nsElement getCMNamespace();

    /**
     * Returns <code>CMNode</code> list of all the constituent nodes in the 
     * content model. 
     * @return List of all <code>CMNodes</code> of the content model.
     */
    public CMNamedNodeMap getCMNodes();

    /**
     * Removes the specified<code>CMNode</code>.
     * @param node <code>CMNode</code> to be removed.
     * @return <code>Success or failure.</code>.
     */
    public boolean removeNode(CMNode node);

    /**
     * Insert <code>CMNode</code>.
     * @param newNode <code>CMNode</code> to be inserted.
     * @param refNode <code>CMNode</code> to be inserted before.
     * @return <code>Success or failure.</code>.
     */
    public boolean insertBefore(CMNode newNode, 
                                CMNode refNode);

    /**
     * Determines if a CMModel and CMExternalModel itself is valid, i.e., 
     * confirming that it's well-formed and valid per its own formal 
     * grammar. Note that within a CMModel, a pointer to a CMExternalModel 
     * can exist.
     * @return Is the CM valid?
     */
    public boolean validate();

    /**
     * Creates an element declaration for the element type specified. The 
     * returned object implements CMElementDeclaration interface. 
     * @param namespaceURI 
     * @param qualifiedElementName The qualified name of the element type 
     *   being declared.
     * @param contentSpec Constant for MIXED, EMPTY, ANY and CHILDREN.
     * @return A new CMElementDeclaration object with <code>name</code> 
     *   attribute set to qualifiedElementName and the 
     *   <code>contentType</code> set to contentSpec. Other attributes of 
     *   the element declaration are set through CMElementDeclaration 
     *   interface methods.
     * @exception DOMException
     *   INVALID_CHARACTER_ERR: Raised if the specified name contains an 
     *   illegal character.
     *   <br>DUPLICATE_NAME_ERR:Raised if an element declaration already 
     *   exists with the same name for a given CMModel.
     */
    public CMElementDeclaration createCMElementDeclaration(String namespaceURI, 
                                                           String qualifiedElementName, 
                                                           int contentSpec)
                                                           throws DOMException;

    /**
     * Creates an attribute declaration. The returned object implements 
     * CMAttributeDeclaration interface. 
     * @param namespaceURI 
     * @param qualifiedName The name of the attribute being declared.
     * @return A new CMAttributeDeclaration object with 
     *   <code>attributeName</code> attribute set to input qualifiedname 
     *   parameter.
     * @exception DOMException
     *   INVALID_CHARACTER_ERR: Raised if the specified name contains an 
     *   illegal character.
     */
    public CMAttributeDeclaration createCMAttributeDeclaration(String namespaceURI, 
                                                               String qualifiedName)
                                                               throws DOMException;

    /**
     * Creates a new notation declaration. The returned object implements 
     * CMNotationDeclaration interface. 
     * @param name The name of the notation being declared.
     * @param systemIdentifier The system identifier for the notation 
     *   declaration.
     * @param publicIdentifier The public identifier for the notation 
     *   declaraiton.
     * @return A new CMNotationDeclaration object with 
     *   <code>notationName</code> attribute set to name.
     * @exception DOMException
     *   INVALID_CHARACTER_ERR: Raised if the specified name contains an 
     *   illegal character.
     *   <br>DUPLICATE_NAME_ERR:Raised if a notation declaration already 
     *   exists with the same name for a given CMModel.
     */
    public CMNotationDeclaration createCMNotationDeclaration(String name, 
                                                             String systemIdentifier, 
                                                             String publicIdentifier)
                                                             throws DOMException;

    /**
     * Creates a new entity declaration. The returned object implements 
     * CMEntityDeclaration interface. 
     * @param name The name of the entity being declared.
     * @return A new CMNotationDeclaration object with <code>entityName</code>
     *    attribute set to name.
     * @exception DOMException
     *   INVALID_CHARACTER_ERR: Raised if the specified name contains an 
     *   illegal character.
     */
    public CMEntityDeclaration createCMEntityDeclaration(String name)
                                                         throws DOMException;

    /**
     * Creates a new CMChildren object. The subModels of the CMChildren is 
     * build using CMChildren interface methods.
     * @param minOccurs The minimum occurance for the subModels of this 
     *   CMChildren.
     * @param maxOccurs The maximum occurance for the subModels of this 
     *   CMChildren.
     * @param operator operator of type CHOICE, SEQ or NONE
     * @return A new CMChildren object.
     * @exception DOMException
     *   INVALID_CHARACTER_ERR: Raised if the specified name contains an 
     *   illegal character.
     */
    public CMChildren createCMChildren(int minOccurs, 
                                       int maxOccurs, 
                                       short operator)
                                       throws DOMException;

}
