/*
 * Copyright (c) 1999 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de Recherche
 *  en Informatique et en Automatique, Keio University).
 * All Rights Reserved. http://www.w3.org/Consortium/Legal/
 */

package org.w3c.dom.range;

import org.w3c.dom.*;

/**
 */
public interface Range {
  /**
   * Node within which the range begins 
   */
  public Node               getStartContainer();
  /**
   * Offset within the starting node of the range. 
   */
  public int                getStartOffset();
  /**
   * Node within which the range ends 
   */
  public Node               getEndContainer();
  /**
   * Offset within the ending node of the range. 
   */
  public int                getEndOffset();
  /**
   * TRUE if the range is collapsed 
   */
  public boolean            getIsCollapsed();
  /**
   * The common ancestor                                  container of the 
   * range's two end-points.
   */
  public Node               getCommonAncestorContainer();
  /**
   * Sets the attributes describing the start of therange. 
   * @param node The <code>startNode</code> value. Thisparameter must be       
   *                                            non-null.
   * @param offset The <code>startOffset</code> value. 
   * @exception RangeException
   *   NULL_NODE_ERR: Raised if <code>node</code>is null.
   *   <br>INVALID_NODE_TYPE_ERR: Raised if<code>node</code> or an            
   *                                       ancestor of <code>node</code> is 
   *   anAttr, Entity, Notation, or DocumentType                              
   *                     node.If an offset is out-of-bounds, shouldit just be 
   *   fixed up                                                  or should an 
   *   exception be raised.
   */
  public void               setStart(Node node, 
                                     int offset)
                                     throws RangeException;
  /**
   * Sets the attributes describing the end of a range.
   * @param node The <code>endNode</code> value. Thisparameter must be         
   *                                          non-null.
   * @param offset The <code>endOffset</code> value. 
   * @exception RangeException
   *   NULL_NODE_ERR: Raised if <code>node</code>is null.
   *   <br>INVALID_NODE_TYPE_ERR: Raised if<code>node</code> or an            
   *                                       ancestor of <code>node</code> is 
   *   anAttr, Entity, Notation, or DocumentType                              
   *                     node.
   */
  public void               setEnd(Node node, 
                                   int offset)
                                   throws RangeException;
  /**
   * Sets the start position to be before a node
   * @param node Range starts before <code>node</code> 
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if an ancestorof <code>node</code> is an 
   *   Attr, Entity,Notation, or DocumentType node or if <code>node</code> is 
   *   a Document,DocumentFragment, Attr, Entity, or Notation                 
   *                                  node.
   */
  public void               setStartBefore(Node node)
                                           throws RangeException;
  /**
   * Sets the start position to be after a node
   * @param node Range starts after <code>node</code> 
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if an ancestorof <code>node</code> is an 
   *   Attr, Entity,Notation, or DocumentType node or if <code>node</code> is 
   *   a Document,DocumentFragment, Attr, Entity, or Notation                 
   *                                  node.
   */
  public void               setStartAfter(Node node)
                                          throws RangeException;
  /**
   * Sets the end position to be before a node. 
   * @param node Range ends before <code>node</code> 
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if an ancestorof <code>node</code> is an 
   *   Attr, Entity,Notation, or DocumentType node or if <code>node</code> is 
   *   a Document,DocumentFragment, Attr, Entity, or Notation                 
   *                                  node.
   */
  public void               setEndBefore(Node node)
                                         throws RangeException;
  /**
   * Sets the end of a range to be after a node 
   * @param node Range ends after <code>node</code>. 
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if an ancestorof <code>node</code> is an 
   *   Attr, Entity,Notation or DocumentType node or if <code>node</code> is 
   *   a Document,DocumentFragment, Attr, Entity, or Notation                 
   *                                  node.
   */
  public void               setEndAfter(Node node)
                                        throws RangeException;
  /**
   * Collapse a range onto one of its end-points 
   * @param toStart If TRUE, collapses the Range onto its start;if FALSE,      
   *                                             collapses it onto its end. 
   */
  public void               collapse(boolean toStart);
  /**
   * Select a node and its contents 
   * @param node The node to select. 
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if an ancestorof <code>node</code> is an 
   *   Attr, Entity,Notation or DocumentType node or if <code>node</code> is 
   *   a Document,DocumentFragment, Attr, Entity, or Notation                 
   *                                  node.
   */
  public void               selectNode(Node node)
                                       throws RangeException;
  /**
   * Select the contents within a node 
   * @param node Node to select from 
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if<code>node</code> or an                
   *                                   ancestor of <code>node</code> is 
   *   anAttr, Entity, Notation or DocumentType                               
   *                    node.
   */
  public void               selectNodeContents(Node node)
                                               throws RangeException;

  public static final int StartToStart = 1;
  public static final int StartToEnd   = 2;
  public static final int EndToEnd     = 3;
  public static final int EndToStart   = 4;


  /**
   * Compare the end-points of two ranges in a document.
   * @param how 
   * @param sourceRange 
   * @return  -1, 0 or 1 depending on whether the correspondingend-point       
   *                                     of the Range is before, equal to, or 
   *   after thecorresponding end-point of <code>sourceRange</code>. 
   * @exception DOMException
   *   WRONG_DOCUMENT_ERR: Raised if the two Rangesare not in the             
   *                                      same document or document fragment.
   */
  public short              compareEndPoints(int how, 
                                             Range sourceRange)
                                             throws DOMException;
  /**
   * Removes the contents of a range from the containingdocument or            
   *                       document fragment without returning a reference to 
   * theremoved content.  
   * @exception DOMException
   *   NO_MODIFICATION_ALLOWED_ERR: Raised if anyportion of the               
   *                                    content of the range is read-only or 
   *   anyof the nodes that contain any of the                                
   *                   content of the range areread-only.
   */
  public void               deleteContents()
                                           throws DOMException;
  /**
   * Moves the contents of a range from the containingdocument or              
   *                     document fragment to a new DocumentFragment. 
   * @return A DocumentFragment containing the extractedcontents. 
   * @exception DOMException
   *   NO_MODIFICATION_ALLOWED_ERR: Raised if anyportion of the               
   *                                    content of the range is read-only or 
   *   anyof the nodes which contain any of the                               
   *                    content of the range are read-only.
   *   <br>HIERARCHY_REQUEST_ERR: Raised if aDocumentType node would          
   *                                         be extracted into the 
   *   newDocumentFragment.
   */
  public DocumentFragment   extractContents()
                                            throws DOMException;
  /**
   * Duplicates the contents of a range 
   * @return A DocumentFragment containing contents equivalentto those of      
   *                                      this range. 
   * @exception DOMException
   *   HIERARCHY_REQUEST_ERR: Raised if aDocumentType node would              
   *                                     be extracted into the 
   *   newDocumentFragment.
   */
  public DocumentFragment   cloneContents()
                                          throws DOMException;
  /**
   * Inserts a node into the document or document fragmentat the               
   *                    start of the range. 
   * @param node The node to insert at the start of therange 
   * @exception DOMException
   *   NO_MODIFICATION_ALLOWED_ERR: Raised if an ancestor container of the 
   *   start                                                of the range is 
   *   read-only.
   *   <br>WRONG_DOCUMENT_ERR: Raised if<code>node</code> and the container 
   *   of the start of the Range were                                         
   *          not created from the same document.
   *   <br>HIERARCHY_REQUEST_ERR: Raised if the container of the start of the 
   *   Range is of                                                a type that 
   *   does not allow children ofthe type of <code>node</code> or if 
   *   <code>node</code> is an ancestor of thecontainer.
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if<code>node</code> is an                
   *                                   Attr, Entity, 
   *   Notation,DocumentFragment, or Document node.
   */
  public void               insertNode(Node node)
                                       throws DOMException, RangeException;
  /**
   * Reparents the contents of the range to the given nodeand                  
   *                 inserts the node at the position of the start of 
   * therange. 
   * @param node The node to surround the contents with. 
   * @exception DOMException
   *   NO_MODIFICATION_ALLOWED_ERR: Raised if an ancestor container of either 
   *                                                  end-point of the range 
   *   is read-only.
   *   <br>WRONG_DOCUMENT_ERR: Raised if<code>node</code> and the container 
   *   of the start of the Range were                                         
   *          not created from the same document.
   *   <br>HIERARCHY_REQUEST_ERR: Raised if the container of the start of the 
   *   Range is of                                                a type that 
   *   does not allow children ofthe type of <code>node</code> or if 
   *   <code>node</code> is an ancestor of thecontaineror if <code>node</code>
   *    would end up                                                with a 
   *   child node of a type not allowedby the type of <code>node</code>.
   * @exception RangeException
   *   BAD_ENDPOINTS_ERR: Raised if the range partially selects a non-text    
   *                                               node.
   *   <br>INVALID_NODE_TYPE_ERR: Raised if<code>node</code> is an            
   *                                       Attr, Entity, DocumentType, 
   *   Notation,Document, or DocumentFragment node.
   */
  public void               surroundContents(Node node)
                                             throws DOMException, RangeException;
  /**
   * Produces a new range whose end-points are equal tothe                     
   *              end-points of the range. 
   * @return The duplicated range. 
   */
  public Range              cloneRange();
  /**
   * Returns the contents of a range as a string. 
   * @return The contents of the range.
   */
  public String             toString();
}

