/*
 * Copyright (c) 1999 World Wide Web Consortium,
 * (Massachusetts Institute of Technology, Institut National de Recherche
 *  en Informatique et en Automatique, Keio University).
 * All Rights Reserved. http://www.w3.org/Consortium/Legal/
 */

package org.w3c.dom.namespaces;

import org.w3c.dom.*;

/**
 * The <code>Element</code> interface is extended to provides a set of methods 
 * to manipulate attributes with namespaces.Both the <code>tagName</code> 
 * attribute from the <code>Element</code> interface and the 
 * <code>nodeName</code> attribute from the <code>Node</code> interface 
 * return the qualified name.
 */
public interface ElementNS {
  /**
   * Retrieves an attribute value by name and namespace name.
   * @param namespaceName The namespace name of the attribute to retrieve.
   * @param localName The local name of the attribute to retrieve.
   * @return The <code>Attr</code> value as a string, or an empty  string if 
   *   that attribute does not have a specified or default value.
   */
  public String             getAttributeNS(String namespaceName, 
                                           String localName);
  /**
   * Adds a new attribute. If an attribute with that local name and namespace 
   * name is already present in the element, its value is changed to be that 
   * of the value parameter. This value is a simple string, it is not parsed 
   * as it is being set. So any markup (such as syntax to be recognized as an 
   * entity reference) is treated as literal text, and needs to be 
   * appropriately escaped by the implementation when it is written out. In 
   * order to assign an attribute value that contains entity references, the 
   * user must create an <code>Attr</code> node plus any <code>Text</code> and
   *  <code>EntityReference</code> nodes, build the appropriate subtree, and 
   * use <code>setAttributeNodeNS</code> or <code>setAttributeNode</code> to 
   * assign it as the value of an attribute.
   * @param namespaceName The namespace name of the attribute to create or 
   *   alter.
   * @param localName The local name of the attribute to create or alter.
   * @param value The value to set in string form.
   * @exception DOMException
   *   INVALID_CHARACTER_ERR: Raised if the specified name contains an 
   *   invalid character.
   *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
   */
  public void               setAttributeNS(String namespaceName, 
                                           String localName, 
                                           String value)
                                           throws DOMException;
  /**
   * Removes an attribute by local name and namespace name. If the removed 
   * attribute has a default value it is immediately replaced.
   * @param namespaceName The namespace name of the attribute to remove.
   * @param localName The local name of the attribute to remove.
   * @exception DOMException
   *   NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
   */
  public void               removeAttributeNS(String namespaceName, 
                                              String localName)
                                              throws DOMException;
  /**
   * Retrieves an <code>Attr</code> node by name and namespace name.
   * @param namespaceName The namespace name of the attribute to retreive.
   * @param localName The local name of the attribute to retrieve.
   * @return The <code>Attr</code> node with the specified attribute local 
   *   name and namespace name or <code>null</code> if there is no such 
   *   attribute.
   */
  public Attr               getAttributeNodeNS(String namespaceName, 
                                               String localName);
  /**
   * Adds a new attribute. If an attribute with that local name and namespace 
   * name is already present in the element, it is replaced by the new one.
   * @param newAttr The <code>Attr</code> node to add to the attribute list.
   * @return If the <code>newAttr</code> attribute replaces an existing 
   *   attribute with the same local name and namespace name, the previously 
   *   existing <code>Attr</code> node is returned, otherwise 
   *   <code>null</code> is returned.
   * @exception DOMException
   *   WRONG_DOCUMENT_ERR: Raised if <code>newAttr</code> was created from a 
   *   different document than the one that created the element.
   *   <br>NO_MODIFICATION_ALLOWED_ERR: Raised if this node is readonly.
   *   <br>INUSE_ATTRIBUTE_ERR: Raised if <code>newAttr</code> is already an 
   *   attribute of another <code>ElementNS</code> object. The DOM user must 
   *   explicitly clone <code>Attr</code> nodes to re-use them in other 
   *   elements.
   */
  public Attr               setAttributeNodeNS(Attr newAttr)
                                               throws DOMException;
  /**
   * Returns a <code>NodeList</code> of all the <code>Element</code>s with a 
   * given local name and namespace name in the order in which they would be 
   * encountered in a preorder traversal of the <code>Document</code> tree, 
   * starting from this node.
   * @param namespaceName The namespace name of the elements to match on. The 
   *   special value "*" matches all namespaces.
   * @param localName The local name of the elements to match on. The special 
   *   value "*" matches all local names.
   * @return A new <code>NodeList</code> object containing all the matched 
   *   <code>Element</code>s.
   */
  public NodeList           getElementsByTagNameNS(String namespaceName, 
                                                   String localName);
}

