/*
 * Copyright (c) 1998 World Wide Web Consortium, (Massachusetts Institute of
 * Technology, Institut National de Recherche en Informatique et en
 * Automatique, Keio University).
 * All Rights Reserved. http://www.w3.org/Consortium/Legal/
 */

package org.w3c.dom.range;

import org.w3c.dom.*;

/**
 */
public interface Range {
  /**
   * Node within which the range begins 
   */
  public Node               getStartParent();
  /**
   * Offset in the starting node of the start of the range. 
   */
  public int                getStartOffset();
  /**
   * Node within which the range ends
   */
  public Node               getEndParent();
  /**
   * Offset in the ending node of the end of the range. 
   */
  public int                getEndOffset();
  /**
   * TRUE if the range is collapsed 
   */
  public boolean            getIsCollapsed();
  /**
   * The common ancestor node of the entire range 
   */
  public Node               getCommonParent();
  /**
   * Sets the attribute values describing the start of the range. 
   * @param parent The <code>startNode</code> value.  This  parameter must be 
   *   non-null.
   * @param offset The <code>startOffset</code> value. 
   * @exception RangeException
   *   NULL_PARENT_ERR: Raised if startNode is null.
   *   <br>INVALID_NODE_TYPE_ERR: Raised if an ancestor of startNode is an 
   *   Attr, Entity, Notation or DocumentType node.
   */
  public void               setStart(Node parent, 
                                     int offset)
                                     throws RangeException;
  /**
   * Sets the attributes describing the end of a range. 
   * @param parent The <code>endNode</code> value.  This parameter must be 
   *   non-null.
   * @param offset The <code>endOffset</code> value. 
   * @exception RangeException
   *   NULL_PARENT_ERR: Raised if endNode is null.
   *   <br>INVALID_NODE_TYPE_ERR: Raised if an ancestor of startNode is an 
   *   Attr, Entity, Notation or DocumentType node.
   */
  public void               setEnd(Node parent, 
                                   int offset)
                                   throws RangeException;
  /**
   * Sets the starting position before a node
   * @param sibling Range starts before this node 
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if an ancestor of sibling is an Attr, 
   *   Entity, Notation or DocumentType node or if sibling itself is a 
   *   Document or DocumentFragment node.
   */
  public void               setStartBefore(Node sibling)
                                           throws RangeException;
  /**
   * Sets the starting position after a node
   * @param sibling Range starts after this node 
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if an ancestor of sibling is an Attr, 
   *   Entity, Notation or DocumentType node or if sibling itself is a 
   *   Document or DocumentFragment node.
   */
  public void               setStartAfter(Node sibling)
                                          throws RangeException;
  /**
   * Sets the ending position of a range to be before a given node. 
   * @param sibling Range ends before this node 
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if an ancestor of sibling is an Attr, 
   *   Entity, Notation or DocumentType node or if sibling itself is a 
   *   Document or DocumentFragment node.
   */
  public void               setEndBefore(Node sibling)
                                         throws RangeException;
  /**
   * Sets the ending position of a range to be after a given node 
   * @param sibling Range ends after this node. 
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if an ancestor of sibling is an Attr, 
   *   Entity, Notation or DocumentType node or if sibling itself is a 
   *   Document or DocumentFragment node.
   */
  public void               setEndAfter(Node sibling)
                                        throws RangeException;
  /**
   * Collapse a range onto one of the end points 
   * @param toStart If TRUE, collapses onto the starting node; if FALSE, 
   *   collapses the range onto the ending node. 
   */
  public void               collapse(boolean toStart);
  /**
   * Select a node and its contents 
   * @param n Node to select from 
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if an ancestor of n is an Attr, Entity, 
   *   Notation or DocumentType node or if n itself is a Document or 
   *   DocumentFragment node.
   */
  public void               selectNode(Node n)
                                       throws RangeException;
  /**
   * Select the contents within a node 
   * @param n Node to select from 
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if an ancestor of n is an Attr, Entity, 
   *   Notation or DocumentType node.
   */
  public void               selectNodeContents(Node n)
                                               throws RangeException;

  public static final int StartToStart = 1;
  public static final int StartToEnd   = 2;
  public static final int EndToEnd     = 3;
  public static final int EndToStart   = 4;


  /**
   * Compare the end-points of two ranges in a document. 
   * @param how 
   * @param sourceRange 
   * @return  -1, 0 or 1 depending on whether the corresponding end-point of 
   *   the Range is less than,equal or greater than the corresponding 
   *   end-point of sourceRange. 
   * @exception DOMException
   *   WRONG_DOCUMENT_ERR: Raised if the two Ranges are not in the same 
   *   document or document fragment.
   */
  public short              compareEndPoints(int how, 
                                             Range sourceRange)
                                             throws DOMException;
  /**
   * Removes the contents of a range from the containing document or document 
   * fragment without returning a reference to the removed content.   
   * @exception DOMException
   *   NO_MODIFICATION_ALLOWED_ERR: Raised if any portion of the content of 
   *   the range is readonly or any of the nodes which contain any of the 
   *   content of the range are readonly.
   */
  public void               deleteContents()
                                           throws DOMException;
  /**
   * Moves the contents of a range from the containing document or document 
   * fragment to a new DocumentFragment. 
   * @return A DocumentFragment containing the extracted contents. 
   * @exception DOMException
   *   NO_MODIFICATION_ALLOWED_ERR: Raised if any portion of the content of 
   *   the range is readonly or any of the nodes which contain any of the 
   *   content of the range are readonly.
   */
  public DocumentFragment   extractContents()
                                            throws DOMException;
  /**
   * Duplicates the contents of a range 
   * @return A DocumentFragment containing contents equivalent  to those of 
   *   this range. 
   */
  public DocumentFragment   cloneContents();
  /**
   * inserts the specified node into the document or document  fragment at the 
   * start end-point of the range. 
   * @param n The node to insert at the start end-point of the range 
   * @exception DOMException
   *   NO_MODIFICATION_ALLOWED_ERR: Raised if the parent or any ancestor of 
   *   the start end-point of the range is readonly.
   * @exception RangeException
   *   INVALID_NODE_TYPE_ERR: Raised if n is an Attr, Entity, Notation, 
   *   DocumentType or Document node.
   */
  public void               insertNode(Node n)
                                       throws DOMException, RangeException;
  /**
   * Reparents the contents of the range to the given node and inserts the 
   * node at the location of the start end-point of the range. 
   * @param n The node to surround the contents with. 
   * @exception DOMException
   *   NO_MODIFICATION_ALLOWED_ERR: Raised if the parent or any ancestor of 
   *   the either end-point of the range is readonly.
   * @exception RangeException
   *   BAD_ENDPOINTS_ERR: Raised if the range only partially contains a node.
   *   <br>INVALID_NODE_TYPE_ERR: Raised if n is an Attr, Entity, 
   *   DocumentType, Notation, Document or DocumentFragment node.
   */
  public void               surroundContents(Node n)
                                             throws DOMException, RangeException;
  /**
   * Produces a new range whose end-points are equal to the end-points of the 
   * range. 
   * @return The duplicated range. 
   */
  public Range              cloneRange();
  /**
   * Returns the contents of a range as a string. 
   * @return The contents of the range.
   */
  public String             toString();
}

