/*
 * (c) COPYRIGHT 1999 World Wide Web Consortium
 * (Massachusetts Institute of Technology, Institut National de Recherche
 *  en Informatique et en Automatique, Keio University).
 * All Rights Reserved. http://www.w3.org/Consortium/Legal/
 *
 * $Id: utils.c,v 1.2 2000/03/08 22:29:07 plehegar Exp $
 */

#include <stdio.h>
#include <string.h>

#include "utils.h"

#define MAX_BUFFER 1024

/* quick n' dirty 
 * no free
 */
char * SAC_CSS2_Condition_ToString(const SAC_Condition * condition)
{
    char buffer[MAX_BUFFER];
    
    if (condition == NULL) {
	return "";
    }
    switch (condition->conditionType) {
    case SAC_AND_CONDITION:
	strcpy(buffer, 
	       SAC_CSS2_Condition_ToString(condition->desc
					   .combinator.firstCondition));
	strcat(buffer, 
	       SAC_CSS2_Condition_ToString(condition->desc
					   .combinator.secondCondition));
	break;
    case SAC_OR_CONDITION:
	/* not supported in CSS2 */
	break;
    case SAC_NEGATIVE_CONDITION:
	/* not supported in CSS2 */
	break;
    case SAC_POSITIONAL_CONDITION:
	/* not supported in CSS2 */
	break;
    case SAC_ATTRIBUTE_CONDITION:
	if (condition->desc.attribute.localName != NULL) {
	    /* namespaceURI not supported */
	    strcpy(buffer, "[");
	    strcat(buffer, condition->desc.attribute.localName);
	    if (condition->desc.attribute.value != NULL) {
		strcat(buffer, "=\"");
		strcat(buffer, condition->desc.attribute.value);
		strcat(buffer, "\"");
	    }
	    strcat(buffer, "]");
	} else {
	    /* not supported in CSS2 */
	}
	break;
    case SAC_ONE_OF_ATTRIBUTE_CONDITION:
	if (condition->desc.attribute.localName != NULL) {
	    /* namespaceURI not supported */
	    strcpy(buffer, "[");
	    strcat(buffer, condition->desc.attribute.localName);
	    if (condition->desc.attribute.value != NULL) {
		strcat(buffer, "~=\"");
		strcat(buffer, condition->desc.attribute.value);
		strcat(buffer, "\"");
	    }
	    strcat(buffer, "]");
	} else {
	    /* not supported in CSS2 */
	}
	break;
    case SAC_BEGIN_HYPHEN_ATTRIBUTE_CONDITION:
	if (condition->desc.attribute.localName != NULL) {
	    /* namespaceURI not supported */
	    strcpy(buffer, "[");
	    strcat(buffer, condition->desc.attribute.localName);
	    if (condition->desc.attribute.value != NULL) {
		strcat(buffer, "|=\"");
		strcat(buffer, condition->desc.attribute.value);
		strcat(buffer, "\"");
	    }
	    strcat(buffer, "]");
	} else {
	    /* not supported in CSS2 */
	}
	break;
    case SAC_CLASS_CONDITION:
	strcpy(buffer, ".");
	strcpy(buffer, condition->desc.attribute.value);
	break;
    case SAC_PSEUDO_CLASS_CONDITION:
	strcpy(buffer, ":");
	strcpy(buffer, condition->desc.attribute.value);
	break;
    case SAC_ID_CONDITION:
	strcpy(buffer, "#");
	strcat(buffer, condition->desc.attribute.value);
	break;
    case SAC_LANG_CONDITION:
	strcpy(buffer, ":lang(");
	strcat(buffer, condition->desc.lang);
	strcat(buffer, ")");
	break;
    case SAC_CONTENT_CONDITION:
	/* not supported in CSS2 */
	break;
    case SAC_ONLY_CHILD_CONDITION:
	/* not supported in CSS2 */
	break;
    default:
	// @@ERROR
	break;
    }
    return strdup(buffer);
}

/* quick n' dirty 
 * no free
 */
char * SAC_CSS2_Selector_ToString(const SAC_Selector * selector)
{
    char buffer[MAX_BUFFER];
    
    if (selector == NULL) {
	return "";
    }
    switch (selector->selectorType) {
    case SAC_CONDITIONAL_SELECTOR:
	strcpy(buffer, 
	       SAC_CSS2_Selector_ToString(selector->desc
					  .conditional.simpleSelector));
	strcat(buffer, 
	       SAC_CSS2_Condition_ToString(selector->desc
					   .conditional.condition));
	break;
    case SAC_ANY_NODE_SELECTOR:
	/* not supported in CSS2 */
	break;
    case SAC_NEGATIVE_SELECTOR:
	/* not supported in CSS2 */
	break;
    case SAC_DESCENDANT_SELECTOR:
	strcpy(buffer, 
	       SAC_CSS2_Selector_ToString(selector->desc
					  .combinator.parentSelector));
	strcat(buffer, " ");
	strcat(buffer, 
	       SAC_CSS2_Selector_ToString(selector->desc
					  .combinator.simpleSelector));
	break;
    case SAC_CHILD_SELECTOR:
	strcpy(buffer, 
	       SAC_CSS2_Selector_ToString(selector->desc
					  .combinator.parentSelector));
	strcat(buffer, " ");
	strcat(buffer, 
	       SAC_CSS2_Selector_ToString(selector->desc
					  .combinator.simpleSelector));
	break;
    case SAC_DIRECT_ADJACENT_SELECTOR:
	strcpy(buffer, 
	       SAC_CSS2_Selector_ToString(selector->desc
					  .combinator.parentSelector));
	strcat(buffer, " + ");
	strcat(buffer, 
	       SAC_CSS2_Selector_ToString(selector->desc
					  .combinator.simpleSelector));
	break;
    case SAC_INDIRECT_ADJACENT_SELECTOR:
	/* not supported in CSS2 */
	break;
    case SAC_ELEMENT_NODE_SELECTOR:
	/* namespaceURI not supported */
	if (selector->desc.element.localName != NULL) {
	    strcpy(buffer, selector->desc.element.localName);
	} else {
	    strcpy(buffer, "*");
	}
	break;
    case SAC_PSEUDO_ELEMENT_SELECTOR:
	strcpy(buffer, ":");
	strcpy(buffer, selector->desc.element.localName);
	break;
    case SAC_TEXT_NODE_SELECTOR:
	/* not supported in CSS2 */
	break;
    case SAC_CDATA_SECTION_NODE_SELECTOR:
	/* not supported in CSS2 */
	break;
    case SAC_COMMENT_NODE_SELECTOR:
	/* not supported in CSS2 */
	break;
    case SAC_PROCESSING_INSTRUCTION_NODE_SELECTOR:
	/* not supported in CSS2 */
	break;
    }
    return strdup(buffer);
}


void checkSimpleSelector(SAC_Selector * selector) {
    if (selector == NULL) {
	return;
    }
    switch (selector->selectorType) {
    case SAC_CONDITIONAL_SELECTOR:
    case SAC_ANY_NODE_SELECTOR:
    case SAC_NEGATIVE_SELECTOR:
    case SAC_ELEMENT_NODE_SELECTOR:
    case SAC_PSEUDO_ELEMENT_SELECTOR:
    case SAC_TEXT_NODE_SELECTOR:
    case SAC_CDATA_SECTION_NODE_SELECTOR:
    case SAC_COMMENT_NODE_SELECTOR:
    case SAC_PROCESSING_INSTRUCTION_NODE_SELECTOR:
	return;
    default:
	fprintf(stderr, "The selector %c is not a simple selector.\n",
		selector->selectorType);
	break;
    }
    return;
}

SAC_Selector * SAC_CreateConditionalSelector(SAC_Selector * simpleSelector,
					     SAC_Condition * condition)
{
    SAC_Selector * result;
    
#ifdef SAC_CHECK
    switch (simpleSelector->selectorType) {
    case SAC_ANY_NODE_SELECTOR:
    case SAC_NEGATIVE_SELECTOR:
    case SAC_ELEMENT_NODE_SELECTOR:
    case SAC_PSEUDO_ELEMENT_SELECTOR:
    case SAC_TEXT_NODE_SELECTOR:
    case SAC_CDATA_SECTION_NODE_SELECTOR:
    case SAC_COMMENT_NODE_SELECTOR:
    case SAC_PROCESSING_INSTRUCTION_NODE_SELECTOR:
	break;
    default:
	fprintf(stderr, "The selector %c is not a simple selector.\n",
		simpleSelector->selectorType);
	break;
    }
#endif
    if ((simpleSelector == NULL) || (condition == NULL)) {
	return NULL;
    }
    
    result = (SAC_Selector *) SAC_MALLOC(sizeof(SAC_Selector));
    if (result == NULL) {
	return NULL;
    }
    
    result->selectorType = SAC_CONDITIONAL_SELECTOR;
    result->desc.conditional.simpleSelector = simpleSelector;
    result->desc.conditional.condition = condition;
    return result;
}

SAC_Selector * SAC_CreateAnyNodeSelector()
{
    SAC_Selector * result = 
	(SAC_Selector *) SAC_MALLOC(sizeof(SAC_Selector));
    if (result == NULL) {
	return NULL;
    }
    
    result->selectorType = SAC_ANY_NODE_SELECTOR;
    return result;
}

SAC_Selector * SAC_CreateNegativeSelector(SAC_Selector * simpleSelector)
{
    SAC_Selector * result;
    if (simpleSelector == NULL) {
	return NULL;
    }
    result = (SAC_Selector *) SAC_MALLOC(sizeof(SAC_Selector));
    if (result == NULL) {
	return NULL;
    }
    
#ifdef SAC_CHECK
    checkSimpleSelector(simpleSelector);
#endif
    
    result->selectorType = SAC_NEGATIVE_SELECTOR;
    result->desc.simpleSelector = simpleSelector;
    return result;
}

PRIVATE 
SAC_Selector * _SAC_CreateCombinatorSelector(SAC_SelectorType selectorType,
					     SAC_Selector * parentSelector,
					     SAC_Selector * simpleSelector)
{
    SAC_Selector * result;
    if ((simpleSelector == NULL) || (parentSelector == NULL)) {
	return NULL;
    }
    result = (SAC_Selector *) SAC_MALLOC(sizeof(SAC_Selector));
    if (result == NULL) {
	return NULL;
    }
    
#ifdef SAC_CHECK
    checkSimpleSelector(simpleSelector);
#endif
    
    result->selectorType = selectorType;
    result->desc.combinator.parentSelector = parentSelector;
    result->desc.combinator.simpleSelector = simpleSelector;
    return result;
}

SAC_Selector * SAC_CreateDescendantSelector(SAC_Selector * parentSelector,
					    SAC_Selector * simpleSelector)
{
    return _SAC_CreateCombinatorSelector(SAC_DESCENDANT_SELECTOR, 
					 parentSelector, simpleSelector);
}

SAC_Selector * SAC_CreateChildSelector(SAC_Selector * parentSelector,
				       SAC_Selector * simpleSelector)
{
    return _SAC_CreateCombinatorSelector(SAC_CHILD_SELECTOR, 
					 parentSelector, simpleSelector);
}

SAC_Selector * 
SAC_CreateDirectAdjacentSelector(SAC_Selector * parentSelector,
				 SAC_Selector * simpleSelector)
{
    return _SAC_CreateCombinatorSelector(SAC_DIRECT_ADJACENT_SELECTOR, 
					 parentSelector, simpleSelector);
}

SAC_Selector * 
SAC_CreateIndirectAdjacentSelector(SAC_Selector * parentSelector,
				   SAC_Selector * simpleSelector)
{
    return _SAC_CreateCombinatorSelector(SAC_INDIRECT_ADJACENT_SELECTOR, 
					 parentSelector, simpleSelector);
}

SAC_Selector * SAC_CreateElementNodeSelector(const SAC_STRING namespaceURI,
					     const SAC_STRING localName)
{
    SAC_STRING ls = NULL;
    SAC_Selector * result = 
	(SAC_Selector *) SAC_MALLOC(sizeof(SAC_Selector));
    if (result == NULL) {
	return NULL;
    }
    
    result->selectorType = SAC_ELEMENT_NODE_SELECTOR;
    result->desc.element.namespaceURI = 
	DUP_SAC_STRING(namespaceURI);
    result->desc.element.localName = 
	DUP_SAC_STRING(localName);
    if ((localName != NULL) && (namespaceURI != NULL)
	&& ((ls = SAC_STRCHR(localName, ':')) != NULL)) {
	ls++;
    }
    result->desc.element.localName = ls;
    return result;
}

SAC_Selector * 
SAC_CreateFirstLinePseudoElementSelector(const SAC_STRING namespaceURI)
{
    SAC_Selector * result = 
	(SAC_Selector *) SAC_MALLOC(sizeof(SAC_Selector));
    if (result == NULL) {
	return NULL;
    }
    
    result->selectorType = SAC_PSEUDO_ELEMENT_SELECTOR;
    result->desc.element.namespaceURI = 
	DUP_SAC_STRING(namespaceURI);
    result->desc.element.localName = "first-line";
    result->desc.element.localName = NULL;
    return result;
}

SAC_Selector * 
SAC_CreateFirstLetterPseudoElementSelector(const SAC_STRING namespaceURI)
{
    SAC_Selector * result = 
	(SAC_Selector *) SAC_MALLOC(sizeof(SAC_Selector));
    if (result == NULL) {
	return NULL;
    }
    
    result->selectorType = SAC_PSEUDO_ELEMENT_SELECTOR;
    result->desc.element.namespaceURI = 
	DUP_SAC_STRING(namespaceURI);
    result->desc.element.localName = "first-letter";
    result->desc.element.localName = NULL;
    return result;
}

SAC_Selector * 
SAC_CreateBeforePseudoElementSelector(const SAC_STRING namespaceURI)
{
    SAC_Selector * result = 
	(SAC_Selector *) SAC_MALLOC(sizeof(SAC_Selector));
    if (result == NULL) {
	return NULL;
    }
    
    result->selectorType = SAC_PSEUDO_ELEMENT_SELECTOR;
    result->desc.element.namespaceURI = 
	DUP_SAC_STRING(namespaceURI);
    result->desc.element.localName = "before";
    result->desc.element.localName = NULL;
    return result;
}

SAC_Selector * 
SAC_CreateAfterPseudoElementSelector(const SAC_STRING namespaceURI)
{
    SAC_Selector * result = 
	(SAC_Selector *) SAC_MALLOC(sizeof(SAC_Selector));
    if (result == NULL) {
	return NULL;
    }
    
    result->selectorType = SAC_PSEUDO_ELEMENT_SELECTOR;
    result->desc.element.namespaceURI = 
	DUP_SAC_STRING(namespaceURI);
    result->desc.element.localName = "after";
    result->desc.element.localName = NULL;
    return result;
}

PRIVATE
SAC_Selector * _SAC_CreateCharacterDataSelector(SAC_SelectorType selectorType,
						const SAC_STRING data)
{
    SAC_Selector * result = 
	(SAC_Selector *) SAC_MALLOC(sizeof(SAC_Selector));
    if (result == NULL) {
	return NULL;
    }
    
    result->selectorType = selectorType;
    result->desc.data =
	DUP_SAC_STRING(data);
    return result;
}

SAC_Selector * SAC_CreateTextNodeSelector(const SAC_STRING data)
{
    return _SAC_CreateCharacterDataSelector(SAC_TEXT_NODE_SELECTOR, data);
}

SAC_Selector * SAC_CreateCDATASectionNodeSelector(const SAC_STRING data)
{
    return _SAC_CreateCharacterDataSelector(SAC_CDATA_SECTION_NODE_SELECTOR, 
					    data);
}

SAC_Selector * SAC_CreateCommentNodeSelector(const SAC_STRING data)
{
    return _SAC_CreateCharacterDataSelector(SAC_COMMENT_NODE_SELECTOR, data);
}

SAC_Selector * 
SAC_CreateProcessingInstructionNodeSelector(const SAC_STRING target,
					    const SAC_STRING data)
{
    SAC_Selector * result = 
	(SAC_Selector *) SAC_MALLOC(sizeof(SAC_Selector));
    if (result == NULL) {
	return NULL;
    }
    
    result->selectorType = SAC_PROCESSING_INSTRUCTION_NODE_SELECTOR;
    result->desc.pi.target = DUP_SAC_STRING(target);
    result->desc.pi.data   = DUP_SAC_STRING(data);
    return result;
}

void SAC_DisposeCondition(SAC_Condition * condition);

void SAC_DisposeSelector(SAC_Selector * selector)
{
    if (selector == NULL) {
	return;
    }
    switch (selector->selectorType) {
    case SAC_CONDITIONAL_SELECTOR:
	SAC_DisposeSelector(selector->desc
			    .conditional.simpleSelector);
	SAC_DisposeCondition(selector->desc.conditional.condition);
	break;
    case SAC_ANY_NODE_SELECTOR:
	break;
    case SAC_NEGATIVE_SELECTOR:
	SAC_DisposeSelector(selector->desc.simpleSelector);
	break;
    case SAC_DESCENDANT_SELECTOR:
    case SAC_CHILD_SELECTOR:
    case SAC_DIRECT_ADJACENT_SELECTOR:
    case SAC_INDIRECT_ADJACENT_SELECTOR:
	SAC_DisposeSelector(selector->desc
			    .combinator.parentSelector);
	SAC_DisposeSelector(selector->desc
			    .combinator.simpleSelector);
	break;
    case SAC_ELEMENT_NODE_SELECTOR:
	DISPOSE_SAC_STRING(selector->desc.element.namespaceURI);
	DISPOSE_SAC_STRING(selector->desc.element.localName);
	break;
    case SAC_PSEUDO_ELEMENT_SELECTOR:
	DISPOSE_SAC_STRING(selector->desc.element.namespaceURI);
	break;
    case SAC_TEXT_NODE_SELECTOR:
    case SAC_CDATA_SECTION_NODE_SELECTOR:
    case SAC_COMMENT_NODE_SELECTOR:
	DISPOSE_SAC_STRING(selector->desc.data);
	break;
    case SAC_PROCESSING_INSTRUCTION_NODE_SELECTOR:
	DISPOSE_SAC_STRING(selector->desc.pi.target);
	DISPOSE_SAC_STRING(selector->desc.pi.data);
	break;
    default:
	// @@ERROR
	break;
    }
    SAC_FREE(selector);
}

SAC_Condition * SAC_CreateAndCondition(SAC_Condition * firstCondition,
				       SAC_Condition * secondCondition)
{
    SAC_Condition * result;
    if ((firstCondition == NULL) || (secondCondition == NULL)) {
	return NULL;
    }
    result = (SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_AND_CONDITION;
    result->desc.combinator.firstCondition = firstCondition;
    result->desc.combinator.secondCondition = secondCondition;
    return result;
}

SAC_Condition * SAC_CreateOrCondition(SAC_Condition * firstCondition,
				      SAC_Condition * secondCondition)
{
    SAC_Condition * result;
    if ((firstCondition == NULL) || (secondCondition == NULL)) {
	return NULL;
    }
    result = (SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_OR_CONDITION;
    result->desc.combinator.firstCondition = firstCondition;
    result->desc.combinator.secondCondition = secondCondition;
    return result;
}

SAC_Condition * SAC_CreateNegativeCondition(SAC_Condition * condition)
{
    SAC_Condition * result;
    if (condition == NULL) {
	return NULL;
    }
    result = (SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_NEGATIVE_CONDITION;
    result->desc.condition = condition;
    return result;
}

SAC_Condition * SAC_CreatePositionalCondition(signed int position, 
					      SAC_Boolean typeNode,
					      SAC_Boolean type)
{
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_POSITIONAL_CONDITION;
    result->desc.position.position = position;
    result->desc.position.typeNode = typeNode;
    result->desc.position.type = type;
    return result;
}

SAC_Condition * SAC_CreateAttributeCondition(const SAC_STRING namespaceURI,
					     const SAC_STRING localName,
					     SAC_Boolean specified,
					     const SAC_STRING value)
{
    SAC_STRING ls  = NULL;
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_ATTRIBUTE_CONDITION;
    result->desc.attribute.namespaceURI = 
	DUP_SAC_STRING(namespaceURI);
    result->desc.attribute.localName= DUP_SAC_STRING(localName);
    if ((localName != NULL) && (namespaceURI != NULL)
	&& ((ls = SAC_STRCHR(localName, ':')) != NULL)) {
	ls++;
    }
    result->desc.attribute.localName = ls;
    result->desc.attribute.specified = specified;
    result->desc.attribute.value     = DUP_SAC_STRING(value);
    return result;
}

SAC_Condition * SAC_CreateOneOfAttributeCondition(const SAC_STRING namespaceURI,
						  const SAC_STRING localName,
						  SAC_Boolean specified,
						  const SAC_STRING value)
{
    SAC_STRING ls  = NULL;
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_ONE_OF_ATTRIBUTE_CONDITION;
    result->desc.attribute.namespaceURI = 
	DUP_SAC_STRING(namespaceURI);
    result->desc.attribute.localName= DUP_SAC_STRING(localName);
    if ((localName != NULL) && (namespaceURI != NULL)
	&& ((ls = SAC_STRCHR(localName, ':')) != NULL)) {
	ls++;
    }
    result->desc.attribute.localName = ls;
    result->desc.attribute.specified    = specified;
    result->desc.attribute.value        = DUP_SAC_STRING(value);
    return result;
}

SAC_Condition * 
SAC_CreateBeginHyphenAttributeCondition(const SAC_STRING namespaceURI,
					const SAC_STRING localName,
					const SAC_Boolean specified,
					const SAC_STRING value)
{
    SAC_STRING ls  = NULL;
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_BEGIN_HYPHEN_ATTRIBUTE_CONDITION;
    result->desc.attribute.namespaceURI = 
	DUP_SAC_STRING(namespaceURI);
    result->desc.attribute.localName= DUP_SAC_STRING(localName);
    if ((localName != NULL) && (namespaceURI != NULL)
	&& ((ls = SAC_STRCHR(localName, ':')) != NULL)) {
	ls++;
    }
    result->desc.attribute.localName = ls;
    result->desc.attribute.specified    = specified;
    result->desc.attribute.value        = DUP_SAC_STRING(value);
    return result;
}

SAC_Condition * SAC_CreateClassCondition(const SAC_STRING className)
{
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_CLASS_CONDITION;
    result->desc.attribute.value = DUP_SAC_STRING(className);
    return result;
}

SAC_Condition * SAC_CreateLinkPseudoClassCondition()
{
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_PSEUDO_CLASS_CONDITION;
    result->desc.attribute.value = "link";
    return result;
}

SAC_Condition * SAC_CreateVisitedPseudoClassCondition()
{
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_PSEUDO_CLASS_CONDITION;
    result->desc.attribute.value = "visited";
    return result;
}

SAC_Condition * SAC_CreateHoverPseudoClassCondition()
{
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_PSEUDO_CLASS_CONDITION;
    result->desc.attribute.value = "hover";
    return result;
}

SAC_Condition * SAC_CreateActivePseudoClassCondition()
{
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_PSEUDO_CLASS_CONDITION;
    result->desc.attribute.value = "active";
    return result;
}

SAC_Condition * SAC_CreateFocusPseudoClassCondition()
{
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_PSEUDO_CLASS_CONDITION;
    result->desc.attribute.value = "focus";
    return result;
}

SAC_Condition * SAC_CreateTargetPseudoClassCondition()
{
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_PSEUDO_CLASS_CONDITION;
    result->desc.attribute.value = "target";
    return result;
}

SAC_Condition * SAC_CreateSubjectPseudoClassCondition()
{
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_PSEUDO_CLASS_CONDITION;
    result->desc.attribute.value = "subject";
    return result;
}

SAC_Condition * SAC_CreateIdCondition(const SAC_STRING id)
{
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_ID_CONDITION;
    result->desc.attribute.value = DUP_SAC_STRING(id);
    return result;
}

SAC_Condition * SAC_CreateLangCondition(const SAC_STRING lang)
{
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_LANG_CONDITION;
    result->desc.lang = DUP_SAC_STRING(lang);
    return result;
}

SAC_Condition * SAC_CreateOnlyChildCondition()
{
    SAC_Condition * result = 
	(SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_ONLY_CHILD_CONDITION;
    return result;
}

SAC_Condition * SAC_CreateContentCondition(const SAC_STRING data)
{
    SAC_Condition * result;
    if (data == NULL) {
	return NULL;
    }
    result = (SAC_Condition *) SAC_MALLOC(sizeof(SAC_Condition));
    if (result == NULL) {
	return NULL;
    }
    
    result->conditionType = SAC_CONTENT_CONDITION;
    result->desc.data = DUP_SAC_STRING(data);
    return result;
}

void SAC_DisposeCondition(SAC_Condition * condition)
{
    if (condition == NULL) {
	return;
    }
    switch (condition->conditionType) {
    case SAC_AND_CONDITION:
    case SAC_OR_CONDITION:
	SAC_DisposeCondition(condition->desc
			     .combinator.firstCondition);
	SAC_DisposeCondition(condition->desc
			     .combinator.secondCondition);
	break;
    case SAC_NEGATIVE_CONDITION:
	SAC_DisposeCondition(condition->desc.condition);
	break;
    case SAC_POSITIONAL_CONDITION:
	break;
    case SAC_ATTRIBUTE_CONDITION:
    case SAC_ONE_OF_ATTRIBUTE_CONDITION:
    case SAC_BEGIN_HYPHEN_ATTRIBUTE_CONDITION:
	DISPOSE_SAC_STRING(condition->desc.attribute.namespaceURI);
	DISPOSE_SAC_STRING(condition->desc.attribute.localName);
	DISPOSE_SAC_STRING(condition->desc.attribute.value);
	break;
    case SAC_CLASS_CONDITION:
	DISPOSE_SAC_STRING(condition->desc.attribute.value);
	break;
    case SAC_PSEUDO_CLASS_CONDITION:	
	break;
    case SAC_ID_CONDITION:
	DISPOSE_SAC_STRING(condition->desc.attribute.value);
	break;
    case SAC_LANG_CONDITION:
	DISPOSE_SAC_STRING(condition->desc.lang);
	break;
    case SAC_CONTENT_CONDITION:
	DISPOSE_SAC_STRING(condition->desc.data);
	break;
    case SAC_ONLY_CHILD_CONDITION:
	break;
    default:
	// @@ERROR
	break;
    }
}

SAC_LexicalUnit * SAC_CreateLUOperatorComma() {
    SAC_LexicalUnit * result = 
	(SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_OPERATOR_COMMA;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUOperatorPlus() {
    SAC_LexicalUnit * result = 
	(SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_OPERATOR_PLUS;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUOperatorMinus() {
    SAC_LexicalUnit * result = 
	(SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_OPERATOR_MINUS;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUOperatorMultiply() {
    SAC_LexicalUnit * result = 
	(SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_OPERATOR_MULTIPLY;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUOperatorSlash() {
    SAC_LexicalUnit * result = 
	(SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_OPERATOR_SLASH;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUOperatorMod() {
    SAC_LexicalUnit * result = 
	(SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_OPERATOR_MOD;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUOperatorExp() {
    SAC_LexicalUnit * result = 
	(SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_OPERATOR_EXP;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUOperatorLT() {
    SAC_LexicalUnit * result = 
	(SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_OPERATOR_LT;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUOperatorGT() {
    SAC_LexicalUnit * result = 
	(SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_OPERATOR_GT;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUOperatorLE() {
    SAC_LexicalUnit * result = 
	(SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_OPERATOR_LE;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUOperatorGE() {
    SAC_LexicalUnit * result = 
	(SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_OPERATOR_GE;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUOperatorTilde() {
    SAC_LexicalUnit * result = 
	(SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_OPERATOR_TILDE;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUInherit() {
    SAC_LexicalUnit * result = 
	(SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_INHERIT;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUDimension(const SAC_STRING unit,
					/* signed */ double sreal) {
    SAC_LexicalUnit * result;
    if (unit == NULL) {
	return NULL;
    }
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_DIMENSION;
    result->desc.dimension.unit = SAC_STRDUP(unit);
    result->desc.dimension.value.sreal = sreal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLULengthEM(/* signed */ double sreal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_LENGTH_EM;
    result->desc.dimension.unit = "em";
    result->desc.dimension.value.sreal = sreal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLULengthEX(/* signed */ double sreal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_LENGTH_EX;
    result->desc.dimension.unit = "ex";
    result->desc.dimension.value.sreal = sreal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLULengthPixel(/* signed */ double sreal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_LENGTH_PIXEL;
    result->desc.dimension.unit = "px";
    result->desc.dimension.value.sreal = sreal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLULengthInch(/* signed */ double sreal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_LENGTH_INCH;
    result->desc.dimension.unit = "in";
    result->desc.dimension.value.sreal = sreal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLULengthCentimeter(/* signed */ double sreal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_LENGTH_CENTIMETER;
    result->desc.dimension.unit = "cm";
    result->desc.dimension.value.sreal = sreal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLULengthMillimeter(/* signed */ double sreal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_LENGTH_MILLIMETER;
    result->desc.dimension.unit = "cm";
    result->desc.dimension.value.sreal = sreal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLULengthPoint(/* signed */ double sreal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_LENGTH_POINT;
    result->desc.dimension.unit = "pt";
    result->desc.dimension.value.sreal = sreal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLULengthPica(/* signed */ double sreal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_LENGTH_PICA;
    result->desc.dimension.unit = "pc";
    result->desc.dimension.value.sreal = sreal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUPercentage(/* signed */ double sreal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_PERCENTAGE;
    result->desc.dimension.unit = "%";
    result->desc.dimension.value.sreal = sreal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUDegree(/* unsigned */ double ureal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_DEGREE;
    result->desc.dimension.unit = "deg";
    result->desc.dimension.value.ureal = ureal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUGradian(/* unsigned */ double ureal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_DEGREE;
    result->desc.dimension.unit = "deg";
    result->desc.dimension.value.ureal = ureal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLURadian(/* unsigned */ double ureal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_RADIAN;
    result->desc.dimension.unit = "rad";
    result->desc.dimension.value.ureal = ureal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUMillisecond(/* unsigned */ double ureal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_MILLISECOND;
    result->desc.dimension.unit = "ms";
    result->desc.dimension.value.ureal = ureal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUSecond(/* unsigned */ double ureal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_SECOND;
    result->desc.dimension.unit = "s";
    result->desc.dimension.value.ureal = ureal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUHertz(/* unsigned */ double ureal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_HERTZ;
    result->desc.dimension.unit = "Hz";
    result->desc.dimension.value.ureal = ureal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUKiloHertz(/* unsigned */ double ureal) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_KILOHERTZ;
    result->desc.dimension.unit = "kHz";
    result->desc.dimension.value.ureal = ureal;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUURI(const SAC_STRING uri) {
    SAC_LexicalUnit * result;
    if (uri == NULL) {
	return NULL;
    }
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_URI;
    result->desc.uri = SAC_STRDUP(uri);
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUCounterFunction(SAC_LexicalUnit * parameters) {
    SAC_LexicalUnit * result;
    if (parameters == NULL) {
	return NULL;
    }
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_COUNTER_FUNCTION;
    result->desc.function.name = "counter";
    result->desc.function.parameters = parameters;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUCountersFunction(SAC_LexicalUnit * parameters) {
    SAC_LexicalUnit * result;
    if (parameters == NULL) {
	return NULL;
    }
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_COUNTERS_FUNCTION;
    result->desc.function.name = "counters";
    result->desc.function.parameters = parameters;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLURGBColor(SAC_LexicalUnit * parameters) {
    SAC_LexicalUnit * result;
    if (parameters == NULL) {
	return NULL;
    }
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_RGBCOLOR;
    result->desc.function.name = "rgb";
    result->desc.function.parameters = parameters;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUAttr(const SAC_STRING attrName) {
    SAC_LexicalUnit * result;
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_ATTR;
    result->desc.attrName = DUP_SAC_STRING(attrName);
    return result;    
}
SAC_LexicalUnit * SAC_CreateLURectFunction(SAC_LexicalUnit * parameters) {
    SAC_LexicalUnit * result;
    if (parameters == NULL) {
	return NULL;
    }
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_RECT_FUNCTION;
    result->desc.function.name = "rect";
    result->desc.function.parameters = parameters;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUFunction(const SAC_STRING name,
				       SAC_LexicalUnit * parameters) {
    SAC_LexicalUnit * result;
    if (name == NULL) {
	return NULL;
    }
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_FUNCTION;
    result->desc.function.name = SAC_STRDUP(name);
    result->desc.function.parameters = parameters;
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUIdent(const SAC_STRING ident) {
    SAC_LexicalUnit * result;
    if (ident == NULL) {
	return NULL;
    }
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_IDENT;
    result->desc.ident = SAC_STRDUP(ident);
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUStringValue(const SAC_STRING stringValue) {
    SAC_LexicalUnit * result;
    if (stringValue == NULL) {
	return NULL;
    }
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_STRING_VALUE;
    result->desc.stringValue = SAC_STRDUP(stringValue);
    return result;    
}
SAC_LexicalUnit * SAC_CreateLUSubExpression(SAC_LexicalUnit * subValues) {
    SAC_LexicalUnit * result;
    if (subValues == NULL) {
	return NULL;
    }
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_SUB_EXPRESSION;
    result->desc.subValues = subValues;
    return result;    
}

/* @@ unicode range */
SAC_LexicalUnit * SAC_CreateLUUnicodeRange(const SAC_STRING unicodeRange) {
    SAC_LexicalUnit * result;
    if (unicodeRange == NULL) {
	return NULL;
    }
    result = (SAC_LexicalUnit *) SAC_MALLOC(sizeof(SAC_LexicalUnit));
    if (result == NULL) {
	return NULL;
    }
    
    result->lexicalUnitType = SAC_UNICODERANGE;
    result->desc.unicodeRange = SAC_STRDUP(unicodeRange);
    return result;    
}

void SAC_DisposeLU(SAC_LexicalUnit * lexicalUnit) {
    if (lexicalUnit == NULL) {
	return;
    }
    switch (lexicalUnit->lexicalUnitType) {
    case SAC_OPERATOR_COMMA:			
    case SAC_OPERATOR_PLUS:			
    case SAC_OPERATOR_MINUS:			
    case SAC_OPERATOR_MULTIPLY:		
    case SAC_OPERATOR_SLASH:			
    case SAC_OPERATOR_MOD:			
    case SAC_OPERATOR_EXP:			
    case SAC_OPERATOR_LT:			
    case SAC_OPERATOR_GT:			
    case SAC_OPERATOR_LE:			
    case SAC_OPERATOR_GE:			
    case SAC_OPERATOR_TILDE:				
    case SAC_INHERIT:			
    case SAC_INTEGER:
    case SAC_REAL:
    case SAC_LENGTH_EM:
    case SAC_LENGTH_EX:
    case SAC_LENGTH_PIXEL:
    case SAC_LENGTH_INCH:
    case SAC_LENGTH_CENTIMETER:
    case SAC_LENGTH_MILLIMETER:
    case SAC_LENGTH_POINT:
    case SAC_LENGTH_PICA:
    case SAC_PERCENTAGE:
    case SAC_DEGREE:
    case SAC_GRADIAN:
    case SAC_RADIAN:	
    case SAC_MILLISECOND:
    case SAC_SECOND:
    case SAC_HERTZ:
    case SAC_KILOHERTZ:
	/* nothing to do */
	break;
    case SAC_URI:
	DISPOSE_SAC_STRING(lexicalUnit->desc.uri);
	break;
    case SAC_COUNTER_FUNCTION:
    case SAC_COUNTERS_FUNCTION:
    case SAC_RGBCOLOR:		
    case SAC_RECT_FUNCTION:		
	SAC_DisposeLU(lexicalUnit->desc.function.parameters);
	break;	
    case SAC_ATTR:		
	DISPOSE_SAC_STRING(lexicalUnit->desc.attrName);
	break;
    case SAC_IDENT:			
	DISPOSE_SAC_STRING(lexicalUnit->desc.ident);
	break;
    case SAC_STRING_VALUE:
	DISPOSE_SAC_STRING(lexicalUnit->desc.stringValue);
	break;
    case SAC_UNICODERANGE:
	DISPOSE_SAC_STRING(lexicalUnit->desc.unicodeRange);
	break;	
    case SAC_SUB_EXPRESSION:		
	SAC_DisposeLU(lexicalUnit->desc.subValues);
	break;	
    case SAC_FUNCTION:		
	DISPOSE_SAC_STRING(lexicalUnit->desc.function.name);
	SAC_DisposeLU(lexicalUnit->desc.function.parameters);
	break;	
    case SAC_DIMENSION:		
	DISPOSE_SAC_STRING(lexicalUnit->desc.dimension.unit);
	break;	
    default:
	/* @@ERROR */
	break;
    }
}
