/* Part of CPP library.  (Macro hash table support.)
   Copyright (C) 1986, 87, 89, 92, 93, 94, 1995 Free Software Foundation, Inc.
   Written by Per Bothner, 1994.
   Based on CCCP program by by Paul Rubin, June 1986
   Adapted to ANSI C, Richard Stallman, Jan 1987

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

 In other words, you are welcome to use, share and improve this program.
 You are forbidden to forbid anyone else to use, share and improve
 what you give them.   Help stamp out software-hoarding!  */

#include "StdAfx.h"
#include "All.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

BOOL IsEqual(const char* s1, const char* s2)
{
	if(s1 == NULL || s2 == NULL) return FALSE;
	return strcmp(s1, s2) == 0;
}

BOOL IsEqual(const xmlChar* s1, const char* s2)
{
	if(s1 == NULL || s2 == NULL) return FALSE;
	return strcmp((const char*)s1, s2) == 0;
}

BOOL IsEqual(const char* s1, const xmlChar* s2)
{
	if(s1 == NULL || s2 == NULL) return FALSE;
	return strcmp(s1, (const char*)s2) == 0;
}

BOOL IsEqual(const xmlChar* s1, const xmlChar* s2)
{
	if(s1 == NULL || s2 == NULL) return FALSE;
	return strcmp((const char*)s1, (const char*)s2) == 0;
}

//
//
//	class P3P_Policy
//
//
P3P_Policy::P3P_Policy(const char* uri)
 : _policyURI(uri), _body(NULL), _xmlDoc(NULL), _lifetime(0)
{
}

P3P_Policy::~P3P_Policy()
{
	xmlFreeDoc(_xmlDoc);
	delete _body;
}

BOOL P3P_Policy::IsValid(time_t time)
{
	//time coverage checking must be done here

	return TRUE;	
}

BOOL P3P_Policy::IsCovering(const char* uri2)
{
	CString uri(uri2);

	if(uri == _apply && _prefix.IsEmpty())
	{
		return TRUE;
	}

	BOOL result = FALSE;
	POSITION pos = _prefix.GetHeadPosition();
	while(pos)
	{
		const char* pre = _prefix.GetNext(pos);
		if(uri.Find(pre) == 0)
		{
			result = TRUE;
			break;
		}
	}
	if(! result) return FALSE;

	pos = _exclude.GetHeadPosition();
	while(pos)
	{
		const char* exc = _exclude.GetNext(pos);
		if(uri.Find(exc) == 0)
		{
			return FALSE;
		}
	}
	
	return TRUE;
}

BOOL P3P_Policy::AsNaturalLang(char* buf, long length)
{
	strncpy(buf, "This is policy", length);
	return TRUE;
}

xmlDoc* P3P_Policy::XMLDoc()
{
	if(! _xmlDoc)
	{
		if(! _body)
		{
			P3P_PolicyManager::GetInstance()->FetchPolicy(this);
		}
		_xmlDoc = xmlParseMemory(_body, strlen(_body));
	}

	return _xmlDoc;
}

void P3P_Policy::ClearInfo()
{
	_apply.Empty();
	_lifetime = 0;
	_exclude.RemoveAll();
	_prefix.RemoveAll();
}

BOOL P3P_Policy::SetApplyURI(const char* uri)
{
	_apply = uri;
	return TRUE;
}

BOOL P3P_Policy::AddPrefix(const char* prefix)
{
	_prefix.AddTail(prefix);
	return TRUE;
}

BOOL P3P_Policy::AddExclude(const char* exclude)
{
	_exclude.AddTail(exclude);
	return TRUE;
}

BOOL P3P_Policy::SetLifeTime(time_t lifetime)
{
	_lifetime = lifetime;
	return TRUE;
}
//
//
//	class P3P_PolicyManager
//
//
P3P_PolicyManager P3P_PolicyManager::_instance;

P3P_PolicyManager* P3P_PolicyManager::GetInstance()
{
	return &_instance;
}

P3P_PolicyManager::~P3P_PolicyManager()
{
	xmlDoc* doc = xmlNewDoc((const xmlChar*)"1.0");
	//<POLICY-CACHE>
    doc->root = xmlNewDocNode(doc, NULL, P3Ps_POLICY_CACHE, NULL);
 
	P3P_Policy* policy;
	char buf[BUF_LEN];
	while(!_policyList.IsEmpty())
	{
		policy = _policyList.RemoveHead();

		if(policy->GetLifeTime() > CTime::GetCurrentTime().GetTime() + P3P_CACHE_MARGIN)
		{

			//<POLICY uri=".." apply=".." time="...">
			xmlNode* pol = xmlNewChild(doc->root, NULL, P3Ps_POLICY, NULL);
			xmlSetProp(pol, P3Ps_URI, (const xmlChar*)policy->GetPolicyURI());
			xmlSetProp(pol, P3Ps_APPLY_URI, (const xmlChar*)policy->GetApplyURI());
			itoa(policy->GetLifeTime(), buf, 10);
			xmlSetProp(pol, P3Ps_LIFETIME, (const xmlChar*)buf);

			//<PREFIX-GROUP>
			xmlNode* node = xmlNewChild(pol, NULL, P3Ps_ALLOW_GROUP, NULL);
			POSITION pos = policy->GetPrefix()->GetHeadPosition();
			while(pos)
			{
				xmlNode* pre = xmlNewChild(node, NULL, P3Ps_ALLOW, NULL);
				xmlSetProp(pre, P3Ps_URI, (const xmlChar*)(const char*)policy->GetPrefix()->GetNext(pos));
			}

			//<EXCLUDE-GROUP>
			node = xmlNewChild(pol, NULL, P3Ps_DENY_GROUP, NULL);
			pos = policy->GetExclude()->GetHeadPosition();
			while(pos)
			{
				xmlNode* pre = xmlNewChild(node, NULL, P3Ps_DENY, NULL);
				xmlSetProp(pre, P3Ps_URI, (const xmlChar*)(const char*)policy->GetExclude()->GetNext(pos));
			}
		}

		delete policy;
	}

	if(!_cacheFile.IsEmpty()) xmlSaveFile(_cacheFile, doc);
	xmlFreeDoc(doc);
	xmlCleanupCharEncodingHandlers();
}

P3P_Policy* P3P_PolicyManager::GetPolicyFromHeader(const char* uri, char* header)
{
	//search `Opt: "http://www.w3.org/P3Pv1"; ns=11`
	P3P_PolicyInfo pinfo[NS_LEN];
	char buf[LINE_LEN+1];
	istrstream myHeader(header);
	int nscount = 0;
	while(! myHeader.eof())
	{
		myHeader.getline(buf, LINE_LEN);
		char* opt = strstr(buf, P3Ps_OPT);
		if(! opt) continue;
		while((opt = strstr(opt, P3Ps_P3P_NS)) != NULL)
		{
			opt += strlen(P3Ps_P3P_NS);
			if(pinfo[nscount].SetNamespace(opt))
			{
				nscount++;
			}
		}
	}
	if(nscount == 0)
	{
		return NULL;
	}

	//search 11-Policy, 11-Prefix, 11-Exclude
	istrstream myHeader2(header);
	while(! myHeader2.eof())
	{
		myHeader2.getline(buf, LINE_LEN);
		int i;
		for(i=0;i<nscount;i++)
		{
			pinfo[i].EatHeaderLine(buf);
		}
	}

	P3P_Policy* policy = SetPolicyInfo(pinfo, nscount, uri, TRUE);
	return policy;
}

P3P_Policy* P3P_PolicyManager::GetPolicyFromHTML(const char* uri, char* contents)
{
	//Not implemented yet
	return NULL;
}

P3P_Policy* P3P_PolicyManager::GetCoveringPolicyFromCache(const char* uri)
{
	POSITION pos = _policyList.GetHeadPosition();
	while(pos)
	{
		P3P_Policy* policy = _policyList.GetNext(pos);
		if(policy->IsCovering(uri))
		{
			return policy;
		}
	}
	return NULL;
}

P3P_Policy* P3P_PolicyManager::LookUp(const char* uri, BOOL fetch)
{
	//not using local disk cache (just to easily write a code)
	POSITION pos = _policyList.GetHeadPosition();
	while(pos)
	{
		P3P_Policy* policy = _policyList.GetNext(pos);
		if(IsEqual(policy->GetPolicyURI(), uri))
		{
			if(policy->GetLifeTime() >= CTime::GetCurrentTime().GetTime() + P3P_CACHE_MARGIN)
			{
				return policy;
			}
		}
	}

	P3P_Policy* policy = new P3P_Policy(uri);
	_policyList.AddTail(policy);
	return policy;
}

BOOL P3P_PolicyManager::FetchPolicy(P3P_Policy* policy)
{
	char* body = ReadFromDiskCache(policy->GetPolicyURI());
	if(! body)
	{
		body = P3P_BrowserDriver::GetFile(policy->GetPolicyURI());
		WriteToDiskCache(policy->GetPolicyURI(), body);
	}
	policy->SetBody(body);
	return TRUE;
}

P3P_Policy* P3P_PolicyManager::SetPolicyInfo(P3P_PolicyInfo* pinfo, int count, const char* uri, BOOL isHeader)
{
	//currentPolicy is a policy which should corresponds with current uri
	P3P_Policy* currentPolicy = NULL;

	int i;
	for(i=0;i<count;i++)
	{
		//creating policy object
		if((! pinfo[i]._policy) || (strlen(pinfo[i]._policy) == 0)) continue;
		CString polURI(pinfo[i]._policy);
		P3P_Policy* policy = LookUp(polURI, FALSE);

		//applied URI
		policy->SetApplyURI(uri);

		//prefix
		char* s1 = pinfo[i]._prefix;
		char* s2;
		while(s1 && (strlen(s1) != 0))
		{
			s2 = strchr(s1, ',');
			if(s2)
			{
				*s2 = 0;
			}
			CString newURI = P3P_Common::AbsoluteURI(uri, s1);
			policy->GetPrefix()->AddTail(newURI);
			if(!s2) break;

			s1=s2+1;
			while(isspace(*s1)) s1++;
		}

		//exclude
		s1 = pinfo[i]._exclude;
		while(s1 && (strlen(s1) != 0))
		{
			s2 = strchr(s1, ',');
			if(s2)
			{
				*s2 = 0;
			}
			CString newURI = P3P_Common::AbsoluteURI(uri, s1);
			policy->GetExclude()->AddTail(newURI);
			if(!s2) break;

			s1=s2+1;
			while(isspace(*s1)) s1++;
		}

		//lifetime
		policy->SetLifeTime(CTime::GetCurrentTime().GetTime() + 86400);
		
		if(policy->IsCovering(uri))
		{
			currentPolicy = policy;
		}
	}

	return currentPolicy;
}

BOOL P3P_PolicyManager::IsVoid(const char* uri)
{
	return _voidList.Find(uri) != NULL;
}

BOOL P3P_PolicyManager::AddAsVoid(const char* uri)
{
	_voidList.AddTail(uri);
	return TRUE;
}

BOOL P3P_PolicyManager::InitializeDiskCache()
{
	char buf[BUF_LEN+1];
	strcpy(buf, P3P_Common::DirName());
	strcat(buf, P3Ps_CACHE);
	_cacheDir = buf;

	BOOL result = TRUE;
	CFileStatus sta;
	if(! CFile::GetStatus(buf, sta))
	{
		result = ::CreateDirectory(buf, NULL);
	}

	if(result)
	{
		strcat(buf, "\\");
		strcat(buf, P3Ps_POLICYCACHE);
		_cacheFile = buf;
		if(!CFile::GetStatus(buf, sta))
		{
			return TRUE;
		}

		xmlDoc* doc = xmlParseFile(buf);
		xmlNode* child = doc->root->childs;
		CString apply, uri;
		time_t lifetime;
		while(child)
		{
			//uri, apply, time
			xmlAttr* attr = child->properties;
			while(attr)
			{
				if(IsEqual(attr->name, P3Ps_URI))
				{
					uri = attr->val->content;
				}
				else if(IsEqual(attr->name, P3Ps_APPLY_URI))
				{
					apply = attr->val->content;
				}
				else if(IsEqual(attr->name, P3Ps_LIFETIME))
				{
					lifetime = atol((const char*)attr->val->content);
				}
				attr = attr->next;
			}

			if(lifetime < CTime::GetCurrentTime().GetTime() + P3P_CACHE_MARGIN)
			{
				child = child->next;
				continue;
			}
			
			P3P_Policy* policy = new P3P_Policy(uri);
			_policyList.AddTail(policy);
			policy->SetApplyURI(apply);
			policy->SetLifeTime(lifetime);

			//prefix
			xmlNode* child2 = child->childs;
			for(xmlNode* child3=child2->childs;child3;child3=child3->next)
			{
				policy->AddPrefix((const char*)child3->properties->val->content);
			}

			//exlucde
			child2 = child2->next;
			for(child3=child2->childs;child3;child3=child3->next)
			{
				policy->AddExclude((const char*)child3->properties->val->content);
			}

			child = child->next;
		}

		xmlFreeDoc(doc);
	}
	
	return result;
}

char* P3P_PolicyManager::ReadFromDiskCache(const char* uri)
{
	char buf[BUF_LEN];
	char* fName = P3P_Common::EncodeURI(uri);
	strcpy(buf, P3P_Common::DirName());
	strcat(buf, P3Ps_CACHE);
	strcat(buf, "\\");
	strcat(buf, fName);

	delete fName;

	CFileStatus sta;
	if(!CFile::GetStatus(buf, sta))
	{
		return NULL;
	}

	CFile file(buf, CFile::modeRead);
	int len = file.GetLength()+1;
	char* body = new char[len];
	len = file.Read(body, len);
	body[len] = 0;
	return body;
}

BOOL P3P_PolicyManager::WriteToDiskCache(const char* uri, const char* body)
{
	char buf[BUF_LEN];
	char* fName = P3P_Common::EncodeURI(uri);
	strcpy(buf, P3P_Common::DirName());
	strcat(buf, P3Ps_CACHE);
	strcat(buf, "\\");
	strcat(buf, fName);
	delete fName;

	CFile file(buf, CFile::modeWrite | CFile::modeCreate);
	file.Write(body, strlen(body));

	return TRUE;
}

//
//
//	P3P_PolicyInfo
//
//
P3P_PolicyInfo::P3P_PolicyInfo()
{
	_ns[0] = 0;
	_policy[0] = 0;
	_prefix[0] = 0;
	_exclude[0] = 0;
	_lifetime[0] = 0;
}

BOOL P3P_PolicyInfo::SetNamespace(char* opt)
{
	int count = 0;
	while(isdigit(*(opt+count)))
	{
		count++;
	}
	if(count >= 2)
	{
		strncpy(_ns, opt, count);
		_ns[count] = '-';
		_ns[count+1] = 0;
		return TRUE;
	}
	return FALSE;
}

void P3P_PolicyInfo::EatHeaderLine(char* buf)
{
	int len = strlen(_ns);
	if(strncmp(buf, _ns, len) != 0)
	{
		return;
	}

	//Now the line is "11-"
	if(strncmp(buf+len, P3Ps_HEAD_POLICY, strlen(P3Ps_HEAD_POLICY)) == 0)
	{
		char* tok = buf + len + strlen(P3Ps_HEAD_POLICY);
		while(isspace(*tok))
		{
			tok++;
		}

		//remove \r or \n from the string end.
		char* endtok = tok + strlen(tok) - 1;
		while(iscntrl(*endtok))
		{
			*endtok = 0;
			endtok--;
		}
		strcpy(_policy, tok);
	}
	else if(strncmp(buf+len, P3Ps_HEAD_PREFIX, strlen(P3Ps_HEAD_PREFIX)) == 0)
	{
		char* tok = buf + len + strlen(P3Ps_HEAD_PREFIX);
		while(isspace(*tok))
		{
			tok++;
		}
		//remove \r or \n from the string end.
		char* endtok = tok + strlen(tok) - 1;
		while(iscntrl(*endtok))
		{
			*endtok = 0;
			endtok--;
		}
		strcpy(_prefix, tok);
	}
	else if(strncmp(buf+len, P3Ps_HEAD_EXCLUDE, strlen(P3Ps_HEAD_EXCLUDE)) == 0)
	{
		char* tok = buf + len + strlen(P3Ps_HEAD_EXCLUDE);
		while(isspace(*tok))
		{
			tok++;
		}
		//remove \r or \n from the string end.
		char* endtok = tok + strlen(tok) - 1;
		while(iscntrl(*endtok))
		{
			*endtok = 0;
			endtok--;
		}
		strcpy(_exclude, tok);
	}
	else if(strncmp(buf+len, P3Ps_HEAD_CACHE, strlen(P3Ps_HEAD_CACHE)) == 0)
	{
		char* tok = buf + len + strlen(P3Ps_HEAD_CACHE);
		while(isspace(*tok))
		{
			tok++;
		}
		//remove \r or \n from the string end.
		char* endtok = tok + strlen(tok) - 1;
		while(iscntrl(*endtok))
		{
			*endtok = 0;
			endtok--;
		}
		strcpy(_lifetime, tok);
	}
}


//
//
//	P3P_Common
//
//
BOOL P3P_Common::Write(const char* section, const char* key, const char* value)
{
	char* fileName = FileName();
	BOOL result = WritePrivateProfileString(section, key, value, fileName);
	delete fileName;
	return result;
}

char* P3P_Common::Read(const char* section, const char* key, const char* defaultValue)
{
	char* fileName = FileName();
	char buf[BUF_LEN+1];
	GetPrivateProfileString(section, key, defaultValue, buf, BUF_LEN, fileName);
	delete fileName;
	return CopyChar(buf);
}

const char* P3P_Common::DirName()
{
	if(_dirName.IsEmpty())
	{
		char buf[BUF_LEN+1];
		if(::GetModuleFileName(g_hInst, buf, BUF_LEN))
		{
			char* pos = strrchr(buf, '\\');
			if(pos)
			{
				*(pos+1) = 0;
			}
			else
			{
				buf[0] = 0;
			}
			_dirName = buf;
			return _dirName;
		}
		ASSERT(0);
		_dirName = "c:/";
	}

	return _dirName;
}

char* P3P_Common::FileName()
{
	char buf[BUF_LEN+1];
	strcpy(buf, DirName());
	strcat(buf, P3Ps_INIFILE);
	return CopyChar(buf);
}

CString P3P_Common::_dirName;

CString P3P_Common::PrintNode(xmlNode* node)
{
	CString str;
	str.Format("<%s", (const char*)node->name);
	xmlAttr* attr = node->properties;
	while(attr)
	{
		str += " ";
		str += (const char*)attr->name;
		str += "=\"";
		str += (const char*)attr->val->content;
		str += "\"";
		attr = attr->next;
	}
	str += ">";
	return str;
}

char* CopyChar(const char* str)
{
	if(!str)
	{
		return NULL;
	}
	else
	{
		char* newStr = new char[strlen(str)+1];
		strcpy(newStr, str);
		return newStr;
	}
}

xmlChar* CopyChar(const xmlChar* xmlstr)
{
	const char* str = (const char*)xmlstr;
	if(!str)
	{
		return NULL;
	}
	else
	{
		char* newStr = new char[strlen(str)+1];
		strcpy(newStr, str);
		return (xmlChar*)newStr;
	}
}

char* P3P_Common::EncodeURI(const char* uri)
{
	int len = strlen(uri);
	char* newURI = new char[len*2+1];

	const char* s1;
	char *s2;
	char rep = '0';
	BOOL needESC = FALSE;
	for(s1=uri,s2=newURI;*s1;s1++,s2++)
	{
		switch(*s1)
		{
			case ':':
				needESC = TRUE;
				rep = '0';
				break;
			case '/':
				needESC = TRUE;
				rep = '1';
				break;
			case '?':
				needESC = TRUE;
				rep = '2';
				break;
			case '_':
				needESC = TRUE;
				rep = '3';
				break;
			case '<':
				needESC = TRUE;
				rep = '4';
				break;
			case '>':
				needESC = TRUE;
				rep = '5';
				break;
			default:
				rep = *s1;
				needESC = FALSE;
		}
		if(needESC)
		{
			*s2 = '_';
			s2++;
		}
		*s2 = rep;
	}
	*s2 = 0;

	return newURI;
}

CString P3P_Common::AbsoluteURI(const char* uri, const char* relative)
{
	// start with "http:" -> absolute itself
	if(strstr(relative, P3Ps_HTTP) == relative)
	{
		return relative;
	}

	// uri does not start with "http://" -> error
	if(strstr(uri, P3Ps_HTTP) != uri)
	{
		ASSERT(0);
		return "";
	}

	char buf[BUF_LEN];
	strcpy(buf, uri);

	// start with '/' -> combine with host name
	if(*relative == '/')
	{
		char* s1 = strchr(buf + strlen(P3Ps_HTTP), '/');
		if(s1)
		{
			// buf is NOT "http://www.w3.org"
			*s1 = 0;
		}
		strcat(buf, relative);
		return buf;
	}

	// combine with dir name
	char* s1 = strrchr(buf, '/');
	s1++;
	if(s1 - buf == (int)strlen(P3Ps_HTTP))
	{
		// buf is "http://www.w3.org"
		strcat(buf, "/");
	}
	else
	{
		// buf is "http://www.w3.org/xxx/yyy"
		*s1 = 0;
	}
	strcat(buf, relative);
	return buf;
}
